/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.forms;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.dao.FormDAO;
import org.ametys.plugins.forms.question.autofill.AbstractStaticAutoFillSource;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Class for adding user directory autofill to SimpleTextQUestionType
 */
public class UDUserAutoFillSource extends AbstractStaticAutoFillSource 
{
    /** Map of ModelItems specific to DefaultStaticAutoFillSource */
    protected Map<String, ModelItem> _defaultAutoFillItems;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    /** The form question ud helper */
    protected FormQuestionUDHelper _formQuestionUDHelper;
    
    /** The form DAO */
    protected FormDAO _formDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _formQuestionUDHelper = (FormQuestionUDHelper) manager.lookup(FormQuestionUDHelper.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _formDAO = (FormDAO) manager.lookup(FormDAO.ROLE);
    }
    
    public String getAutofillValue(FormQuestion question)
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        return (String) _formQuestionUDHelper.getUserContentAttributeValue(question, userIdentity);
    }

    public Map<String, ModelItem> getModelItems()
    {
        _defaultAutoFillItems = _formQuestionUDHelper.getUDModelItems(List.of(ModelItemTypeConstants.STRING_TYPE_ID));
        return _defaultAutoFillItems;
    }

    public List<ViewItem> getViewElements()
    {
        List<ViewItem> viewElements = new ArrayList<>();
        
        for (String attributeId : _defaultAutoFillItems.keySet())
        {
            ViewElement userSelect = new ViewElement();
            userSelect.setDefinition((ElementDefinition< ? >) _defaultAutoFillItems.get(attributeId));
            viewElements.add(userSelect);
        }
        
        return viewElements;
    }

    @Override
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        User user = userIdentity != null ? _userManager.getUser(userIdentity) : null;
        String autofill = user != null ? getAutofillValue(question) : null;
        if (StringUtils.isNotBlank(autofill))
        {
            XMLUtils.createElement(contentHandler, "default-value", autofill);
        }
    }
    
    public boolean isCacheable(FormQuestion question)
    {
        return false;
    }
}
