/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.observation;

import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.cachepolicy.InsertUserServiceCachePolicy;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.cache.CacheInvalidationPolicy;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.Site;

/**
 * {@link Observer} for observing user content validation, for content that are displayed, in order to invalidate cache on front-office.
 */
public class InvalidateCacheOnDisplayedContentValidationObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    /** Cache invalidation policy */
    protected CacheInvalidationPolicy _cachePolicy;
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _cachePolicy = (CacheInvalidationPolicy) manager.lookup(CacheInvalidationPolicy.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        String eventId = event.getId();
        // If event is validation
        if (eventId.equals(ObservationConstants.EVENT_CONTENT_VALIDATED) || eventId.equals(ObservationConstants.EVENT_CONTENT_DELETED))
        {
            // And if the event is about a user, then it is supported
            Object object = event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            if (object != null && object instanceof Content content)
            {
                return _contentTypesHelper.isInstanceOf(content, UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE);
            }
        }
        
        // If the event is not a validation or the content is not a user, the event is not supported
        return false;
    }
    
    /**
     * Get the sites where the user is displayed
     * @param contentId The Id of the user content
     * @return The list of sites
     */
    protected Set<Site> _getSitesWhereUserContentIsDisplayed(String contentId)
    {
        // Retrieve the zone item that are displaying a user content
        String xpathQuery = "//element(*, ametys:zoneItem)[@ametys-internal:service='" + InsertUserServiceCachePolicy.SERVICE_INSERT_USER_ID + "' and ametys:service_parameters/@ametys:contentId='" + contentId + "']";
        
        AmetysObjectIterable<ZoneItem> zoneItems = _resolver.query(xpathQuery);
        
        return zoneItems.stream()
                        .filter(Objects::nonNull)
                        .map(zoneItem -> zoneItem.getZone().getSitemapElement())
                        .filter(sitemapElem -> sitemapElem instanceof Page)
                        .map(page -> page.getSite())
                        .collect(Collectors.toSet());
    }
    
    @Override
    public int getPriority()
    {
        // Will be processed after indexation
        return MAX_PRIORITY + 4000;
    }
    
    @Override
    public void observe(Event event, Map<String, Object> transientVars)
    {
        String eventId = event.getId();
        if (eventId.equals(ObservationConstants.EVENT_CONTENT_VALIDATED))
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);

            Set<Site>  sites = _getSitesWhereUserContentIsDisplayed(content.getId());
            for (Site site : sites)
            {
                try
                {
                    _cachePolicy.invalidateCacheOnContentModification(site, content);
                }
                catch (Exception e)
                {
                    getLogger().error("Unable to invalidate cache for displayed user content " + content, e);
                }
            }
        }
        else if (eventId.equals(ObservationConstants.EVENT_CONTENT_DELETED))
        {
            String contentId = (String) event.getArguments().get(ObservationConstants.ARGS_CONTENT_ID);
            Set<Site>  sites = _getSitesWhereUserContentIsDisplayed(contentId);
            for (Site site : sites)
            {
                try
                {
                    _cachePolicy.invalidateCacheOnContentDeletion(site, contentId);
                }
                catch (Exception e)
                {
                    getLogger().error("Unable to invalidate cache for displayed user content " + contentId, e);
                }
            }
        }
    }
}
