/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.observation;

import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.contentio.synchronize.observation.ObservationConstants;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.web.indexing.solr.SolrPageIndexer;
import org.ametys.web.repository.page.Page;

/**
 * {@link Observer} after synchronization of the user collection
 *
 */
public class UserContentCollectionSynchronizedObserver implements Observer, Serviceable
{
    private UserDirectoryPageHandler _userPageHandler;
    private ContentTypesHelper _contentTypeHelper;
    private SolrPageIndexer _solrPageIndexer;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _userPageHandler = (UserDirectoryPageHandler) manager.lookup(UserDirectoryPageHandler.ROLE);
        _contentTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _solrPageIndexer = (SolrPageIndexer) manager.lookup(SolrPageIndexer.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_CONTENT_COLLECTION_SYNCHRONIZED);
    }

    public int getPriority()
    {
        return MAX_PRIORITY;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        String cTypeId = (String) event.getArguments().get(ObservationConstants.ARGS_COLLECTION_CONTENT_TYPE);
        
        if (_contentTypeHelper.getAncestors(cTypeId).contains(UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE))
        {
            Set<Page> rootPages = _userPageHandler.getUserDirectoryRootPages(cTypeId);
            
            for (Page rootPage : rootPages)
            {
                _userPageHandler.clearCache(rootPage);
                _solrPageIndexer.reindexPage(rootPage.getId(), true, false);
            }
            
        }
    }

}
