/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
var AmetysOrganizationChart = {
    /** Object containing widths for each organization unit */
    _orgunitWidths: {},
    
    /** Object containing width of each chart to not process it if width hasn't changed */ 
    _orgchartsWidth: {},
    
    /**
     * Process algorithm on resized 
     * @param {Array} entries Array containing resized chart
     */
    processChartFromResizeObserver: function(entries)
    {
        var entry = entries[0];
        var chart = $j(entries[0].target);

        var oldChartWidth = AmetysOrganizationChart._orgchartsWidth[chart.attr("id")];
        var newChartWidth = entry.contentRect.width;

        if (oldChartWidth && oldChartWidth!= newChartWidth)
        {
            AmetysOrganizationChart.processChart(chart);
        }
        AmetysOrganizationChart._orgchartsWidth[chart.attr("id")] = newChartWidth;
    },
    
    /**
     * Process algorithm on given chart to display each level in line or column and give space to each unit due to its weight
     * @param {JQuery} chart JQuery object of the chart to process
     */
    processChart: function(chart)
    {
        AmetysOrganizationChart._chart = chart;
        
        // Remove the potential mobile mode class
        AmetysOrganizationChart._chart.removeClass("mobile-mode");
        
        // Display all levels in line for them to take their width
        $j('ul', AmetysOrganizationChart._chart).addClass("display-as-line");
        
        // Measure all visible orgunits
        AmetysOrganizationChart._measureVisibleOrgunits();
        
        // Remove all max-widths on units
        AmetysOrganizationChart.removeMaxWidths();
        
        // Recursively process each level to dertermine wich one will be displayed as line or column
        AmetysOrganizationChart._processLevelDisplay(AmetysOrganizationChart._chart);
        
        // Recursively process each level to compute a max-width
        AmetysOrganizationChart._addMaxWidths();
        
        // Shwitch to mobile mode if all levels are in column
        AmetysOrganizationChart._switchToMobileMode();
    },
    
    /**
     * Measure widths of all visible organization units and keep these measurments
     */
    _measureVisibleOrgunits: function()
    {
        var orgunits = $j('> ul > li, input:checked ~ ul > li', AmetysOrganizationChart._chart);
        var idPrefix = AmetysOrganizationChart._chart.attr("id").replace("chart", "unit") + "-";

        orgunits.each(function() {
            // Get the id or generate it
            var id = $j(this).attr("id");
            
            if (!AmetysOrganizationChart._orgunitWidths[id])
            {
                // Measure the unit
	            var orgunitContent = $j("> div", $j(this));
	            var orgunitPadding = parseInt($j(this).css("padding-left"), 10);
	            orgunitPadding += parseInt($j(this).css("padding-right"), 10);
	            var orgunitMinWidth = parseInt(orgunitContent.css("min-width"), 10) + orgunitPadding;
	            var orgunitMaxWidth = parseInt(orgunitContent.css("max-width"), 10) + orgunitPadding;
	            
	            // Keep measurement
	            AmetysOrganizationChart._orgunitWidths[id] = {
	                minWidth: orgunitMinWidth,
	                maxWidth: orgunitMaxWidth,
	                padding: orgunitPadding
	            }
            }
        });
    },
    
    /**
     * Remove max-width on all oranization units
     */
    removeMaxWidths: function()
    {
        var orgunits = $j('li', AmetysOrganizationChart._chart);
        
        orgunits.each(function() {
            $j(this).css("maxWidth", "none");
        });
    },
    
    /**
     * Process a level to display it as line or column
     * Add width rate to give each unit the space he need to pisplay its children
     * @param {JQuery} node JQuery object containing the level to process
     */
    _processLevelDisplay: function(node)
    {
        // If node has no children, do nothing
        var list = $j('> ul', node);
        if (list.length == 0)
        {
            return;
        }
        
        // Compute available width
        var availableWidth = list.width();
        
        var children = list.children("li");
        
        // Compute all children possible widths
        var childrenPossibleWidths = [];
        var childrenIndex = [];
        children.each(function() {
            childrenPossibleWidths.push(AmetysOrganizationChart._computePossibleWidths($j(this)));
            childrenIndex.push(0);
        });
        
        // Get best widths from all possibilities and sum of it
        var childrenBestWidths = AmetysOrganizationChart._getBestWidths(childrenPossibleWidths, childrenIndex, availableWidth);
        
        if (childrenBestWidths.length <=0)
        {
            // If there is no combination that allow as line display, remove class to display the list as column
            list.removeClass("display-as-line");
        }
        else
        {
            var totalWidth = childrenBestWidths.reduce(function(a, b) { return a + b}, 0);
            // Set best width to each child
            children.each(function(i) {
                var widthRate = childrenBestWidths[i]*100 / totalWidth;
                $j(this).css("width", widthRate + "%");
            });
        }

        // Recurse only on visible children
        children.each(function() {
            if ($j("> input:checked", $j(this)).length > 0)
            {
                AmetysOrganizationChart._processLevelDisplay($j(this));
            }
        });
    },
    
    /**
     * Compute all possible widths for a node
     * @param {JQuery} node JQuery object of the node
     */
    _computePossibleWidths: function(node)
    {
        nodeSizes = AmetysOrganizationChart._orgunitWidths[node.attr("id")];
        var nodeWidth = nodeSizes.minWidth;
        
        var list = $j('> ul', node);
        if ($j("> input:not(:checked)", node).length > 0 || list.length == 0)
        {
            // If node is not expanded or has no children, the only possible width is it own.
            return [nodeWidth];
        }
        
        var children = list.children("li");
        
        var maximumWidthIfAllSubLevelesAreInColumns = nodeWidth;
        var allWidths = [];
        var childrenPossibleWidths = [];
        children.each(function() {
            // Recurse on children
            var childPossibleWidths = AmetysOrganizationChart._computePossibleWidths($j(this));
            childrenPossibleWidths.push(childPossibleWidths);
            // Search for the min-width needed in column mode
            maximumWidthIfAllSubLevelesAreInColumns = Math.max(maximumWidthIfAllSubLevelesAreInColumns, AmetysOrganizationChart._getMinValueOfArray(childPossibleWidths));
        });
        
        allWidths.push(maximumWidthIfAllSubLevelesAreInColumns);
        
        // 
        var allCombinations = AmetysOrganizationChart._computeAllCombinations(childrenPossibleWidths);
        var nodePadding = nodeSizes.padding;
        allCombinations.forEach(function(combination) {
            combination += nodePadding;
            allWidths.push(Math.max(nodeWidth, combination));
        });
        
        return allWidths;
    },
    
    /**
     * Get the minimum value of the given array
     * @param {Array} array the array
     * @return the minimum value of the given array
     */
    _getMinValueOfArray: function(array)
    {
    	var min = -1;
    	for (var i = 0; i < array.length; i++)
		{
    		var width = array[i];
    		if (min == -1 || width < min)
			{
    			min = width;
			}
		}
    	
    	return min;
    },
    
    /**
     * Compute all combinations from given widths
     * @param {Array} possibleWidths Array of possible widths to combinate
     * @return {Array} an array the sums for each combination
     */
    _computeAllCombinations: function(possibleWidths)
    {
        var allCombinations = [];
        
        var uniqueCombinations = AmetysOrganizationChart._getAllUniqueCombinations(possibleWidths);

        // Init indexes array
        var indexes = [];
        for (var i in uniqueCombinations)
    	{
        	indexes.push(0);
    	}
        
        do
        {
        	allCombinations.push(AmetysOrganizationChart._computeSumCombinations(uniqueCombinations, indexes));
        }
        while (AmetysOrganizationChart._incrementIndex(uniqueCombinations, indexes)); // Increment indexes until all combinations are done
        
        return allCombinations;
    },
    
    /**
     * Get all unique possible combinations from possible widths
     * @param {Array} possibleWidths Array of possible widths to combinate
     * @return all unique unique possible combinations
     */
    _getAllUniqueCombinations: function(possibleWidths)
    {
    	var combinations = [];
    	for (var i = 0; i < possibleWidths.length; i++)
    	{
    		combinations.push(Array.from(new Set(possibleWidths[i])));
    	}
    	
    	return combinations;
    },
    
    /**
     * Compute sum of one combination from given widths
     * @param {Array} possibleWidths Array of possible widths to combinate
     * @param {Array} indexes Array of indexes to increment to compute all combinations
     * @return {Long} the sums for one combination
     */
    _computeSumCombinations: function(possibleWidths, indexes)
    {
    	var sum = 0;
        // Compute the sum of current combination
        for (var i = 0; i < possibleWidths.length; i++)
        {
            var widths = possibleWidths[i];
            var index = indexes[i];
            
            sum += widths[index];
        }
        
        return sum;
    },
    
    /**
     * Get the best combination due to available width
     * The best combination his the one which takes the more space but doesn't exceed the available width
     * @param {Array} possibleWidths all combinations
     * @param {Array} indexes Array of indexes to increment to compute all combinations
     * @param {long} availableWidth available width
     * @return {Array} The best combination
     */
    _getBestWidths: function(possibleWidths, indexes, availableWidth)
    {
        var bestWidths = [];
        var bestSum = -1;
        do
        {
            var currentWidths = [];
            var sum = 0;
            // Compute current combination
            for (var i = 0; i < possibleWidths.length; i++)
            {
                var widths = possibleWidths[i];
                var index = indexes[i];
                
                currentWidths.push(widths[index]);
                sum += widths[index];
            }
            
            // Register current combination as the best if it doesn't exceed the available width
            // and if it is a better combination than the already registered one. 
            if (sum <= availableWidth && (bestWidths.length <= 0 || sum > bestSum))
            {
                bestWidths = currentWidths;
                bestSum = sum;
            }
        }
        // Increment indexes to parse all possibilities 
        while (AmetysOrganizationChart._incrementIndex(possibleWidths, indexes));
        
        return bestWidths;
    },
    
    /**
     * Increment indexes to parse every combination
     * @param {Array} array Array with combinations
     * @param {Array} indexes Array with indexes to increment
     * @return {Boolean} true if the indexes have change, false otherwise
     */
    _incrementIndex: function(array, indexes)
    {
        // We use a reversed loop to first increment right indexes and then give advantage to left elements 
        for (var i = indexes.length - 1; i >= 0 ; i--)
        {
            if (indexes[i] + 1 < array[i].length)
            {
                indexes[i]++;
                return true;
            }
            else
            {
                indexes[i] = 0;
            }
        }
        
        return false;

    },
    
    /**
     * Add max-width on organization units
     */
    _addMaxWidths: function()
    {
        var roots = $j('> ul > li', AmetysOrganizationChart._chart);
        roots.each(function() {
            AmetysOrganizationChart._addMaxWidthOnOrgunit($j(this));
        });
    },
    
    /**
     * Add a max-width on the given organization unit
     * @param {JQuery} orgunit JQuery object containing the organization unit
     * @return {long} the max-width setted to the organization unit
     */
    _addMaxWidthOnOrgunit: function(orgunit)
    {
        var nodeMaxWidth = AmetysOrganizationChart._orgunitWidths[orgunit.attr("id")].maxWidth;
        
        var list = $j('> ul', orgunit);
        if ($j("> input:not(:checked)", orgunit).length > 0 || list.length == 0)
        {
            // If orgunit is not expanded or has no children, the max-width is it own.
            orgunit.css("maxWidth", nodeMaxWidth + "px");
            return nodeMaxWidth;
        }
        
        var children = list.children("li");
        
        // Add and get all children max-width
        var childrenMaxWidths = [];
        children.each(function() {
            childrenMaxWidths.push(AmetysOrganizationChart._addMaxWidthOnOrgunit($j(this)));
        });
        
        var finalMaxWidth;
        if (list.hasClass("display-as-line"))
        {
            // In line display, the max-width is the sum off all children max-widths (or the max-width parent if it is really big)
            var childrenMaxWidth = childrenMaxWidths.reduce(function(a, b) { return a + b}, 0);
            finalMaxWidth = Math.max(nodeMaxWidth, childrenMaxWidth);
        }
        else
        {
            // In column display, the max-width is the max between the max-width of parent and each child
            finalMaxWidth = Math.max(nodeMaxWidth, Math.max.apply(null, childrenMaxWidths));
        }
        
        orgunit.css("maxWidth", finalMaxWidth + "px");
        return finalMaxWidth;
    },
    
    /**
     * Switch to mobile mode if all level are displayed in columns
     */
    _switchToMobileMode: function ()
    {
        var onlyColumns = AmetysOrganizationChart._areAllLevelsDisplayedInColumn()
        if (onlyColumns)
        {
            // Remove class from lists that are displayed in line but have only one element
            $j('ul', AmetysOrganizationChart._chart).removeClass("display-as-line");
            
            // Add a mobile mode class
            AmetysOrganizationChart._chart.addClass("mobile-mode");
        }
    },
    
    /**
     * Check if all level are displayed in columns
     * If some levels are displayed in line but have only one child, it is like they are displayed in column
     */
    _areAllLevelsDisplayedInColumn: function()
    {
        var lists = $j('> ul, input:checked ~ ul', AmetysOrganizationChart._chart);
        if (lists.length == 0)
        {
            return false;
        }
        
        var result = true
        
        lists.each(function() {
            if ($j(this).hasClass("display-as-line"))
            {
                var children = $j(this).children("li");
                if (children.length > 1)
                {
                    result = false;
                }
            }
        });
        
        return result;
    },
    
    /**
     * Put the focus on the root of the complete chart
     * @param {JQuery} The current chart
     */
    _goToMainChart: function(currentChart)
    {
        // Destroy focus chart
        $j(".chart-for-focus", currentChart).remove();

        var ulDiv = $j(".chart-no-focus", currentChart);
        // Show complete chart
        ulDiv.show();
        
        // Process chart to calcule width ...
        AmetysOrganizationChart.processChart(ulDiv.parent());
        
        $j(window).scrollTop(currentChart.offset().top);
    },
    
    /**
     * Put the focus on the current div
     * @param {JQuery} The current div
     */
    _goToOrgUnitDiv: function(currentDiv)
    {
    	var currentChart = currentDiv.parents(".organization-chart");

        // Get the first ul div of the complete chart then hide it
        var ulDiv = $j(".chart-no-focus", currentChart);
        ulDiv.hide();
        
        // Move the current selection to the focus ul div
        $j(".chart-for-focus", currentChart).remove();
    	currentChart.append("<ul class='chart-for-focus'/>");
    	var ulDivForFocus = $j(".chart-for-focus", currentChart);

        var liDiv = currentDiv.parents("li:first", currentChart);
        var newLi = liDiv.clone();
        ulDivForFocus.prepend(newLi);
        
        // Hide focus button
        $j("a.focus:first", newLi).hide();
        
        // Show return to parent button
        $j("a.back-to-parent:first", newLi).show();
        
        // Show the link to remove the focus
        $j("a.remove-focus:first", newLi).show();

        // Show new chart
        ulDivForFocus.show();
        
        // Detach the complete chart div to process chart of the new chart
        var detachUlDiv = ulDiv.detach();
        
        // Process chart to calcule width ...
        AmetysOrganizationChart.processChart(ulDivForFocus.parent());
        
        // Attach the complete chart div
        currentChart.append(detachUlDiv);
        
        $j(window).scrollTop(currentChart.offset().top);
    },
    
    /**
     * Put the focus on the parent of the current div
     * @param {DOM} The current div
     */
    _backToParent: function(currentDiv)
    {
    	// Get current chart
    	var currentJqueryDiv = $j(currentDiv);
    	var currentChart = currentJqueryDiv.parents(".organization-chart");
    	
        // Get li div parent
        var liId = currentJqueryDiv.parents("li:first").attr("id");
    	var liDiv = $j("#" + liId, $j(".chart-no-focus", currentChart));
        var liDivParent = liDiv.parents("li:first");
        
        // If the parent div if the root, just remove focus
        if (liDivParent.parent().hasClass("chart-no-focus"))
        {
        	AmetysOrganizationChart._removeFocus();
        }
        else
        {
            window.location.hash = "#" + $j("a.back-to-parent", liDivParent).attr("data-orgunit-id");
        }
        
        $j(window).scrollTop(currentChart.offset().top);
    },
    
    /**
     * Put the currentDiv id on the hash
     * @param {DOM} The current div
     */
    _focusOrgunit: function(currentDiv)
    {
        // Get id of current selection
        window.location.hash = "#" + currentDiv.getAttribute("data-orgunit-id");
    },
    
    /**
     * Clear the hash
     */
    _removeFocus: function()
    {
        window.location.hash = "#udroot";
    }
    
}

$j(document).ready(function() {
    
    /**
     * Process all charts of the document
     */
    function processAllCharts()
    {
        $j(".organization-chart").each(function() {
            AmetysOrganizationChart.processChart($j(this));
        });
    }
    
    /**
     * Add an id on each chart and on each organization unit
     */
    function addIdsOnCharts()
    {
        var usedIds = [];
        
        // Register all existing chart ids to be sure to not use it for another chart
        $j(".organization-chart").each(function() {
            var id = $j(this).attr("id");
            if (id)
            {
                usedIds.push(id);
            }
        });
        
        // Generate ids for charts that don't have one
        $j(".organization-chart").each(function() {
            var id = $j(this).attr("id");
            var idPrefix = "orgchart-";
            if (!id)
            {
                do
                {
                    var randomPart = getRandomInt(1, 1000);
                    id = idPrefix + randomPart;
                }
                // Generated a new id while the current one is already used
                while (usedIds.indexOf(id) >= 0);
                
                $j(this).attr("id", id);
                usedIds.push(id);
            }
            
            // Add an id on each organization unit
            addIdOnUnits($j(this));
        });
    }
    
    /**
     * Add an id to each organization unit of the chart if necessary
     * @param {JQuery} chart Jquery object representing the chart
     */
    function addIdOnUnits(chart)
    {
        var orgunits = $j('li', chart);
        var idPrefix = chart.attr("id").replace("chart", "unit") + "-";
        var currentId = getRandomInt(1, 100);

        orgunits.each(function() {
            // Get the id or generate it
            var id = $j(this).attr("id");
            if (!id)
            {
                id = idPrefix + currentId;
                $j(this).attr("id", id);
                currentId ++;
            }
        });
    }
    
    /**
     * Get a random int between min (included) and max (excluded) 
     * @param {long} min minimum included boundary
     * @param {long} max maximum excluded boundary
     * @return {long} a random integer between geven boundaries
     */
    function getRandomInt(min, max) {
        min = Math.ceil(min);
        max = Math.floor(max);
        return Math.floor(Math.random() * (max - min)) + min;
    }
    
    function init()
    {
    	// Add ids on each chart if necessary
    	addIdsOnCharts();
    	
    	// Add event on resize to process charts
    	if (typeof ResizeObserver !== 'undefined')
    	{
    		var resizeOberver = new ResizeObserver(AmetysOrganizationChart.processChartFromResizeObserver);
    		$j(".organization-chart").each(function() {
    			resizeOberver.observe($j(this).get(0));
    		});
    	}
    	else
    	{
    		$j(window).resize(processAllCharts);
    	}
    	
    	// Add event to process a chart when an organization unit is expanded or collapsed
    	$j(".organization-chart :checkbox").change(function() {
    		var chart = $j(this).parents(".organization-chart");
    		AmetysOrganizationChart.processChart(chart);
    	});
    	
    	processAllCharts();
    	
    	if (location.hash != '' && location.hash.startsWith("#ud-"))
        {
            var id = window.location.hash.substring(1);
            var rootDiv = $j("[data-orgunit-id='" + id + "']");
            if (rootDiv.length > 0)
            {
            	$j(".chart-no-focus").each(function() {
            		$j(this).hide();
            	});
            }
        }
    	
    	$j(window).bind( 'hashchange', function(e) 
	    {
	        if (location.hash != '' && location.hash.startsWith("#ud-"))
	        {
	            var id = window.location.hash.substring(1);
	            var rootDiv = $j("[data-orgunit-id='" + id + "']");
	            if (rootDiv.length > 0)
	            {
	            	$j(".organization-chart").each(function() {
		        		AmetysOrganizationChart._goToOrgUnitDiv($j(rootDiv, $j(this)));
	            	});
	            }
	            else
	            {
	            	$j(".organization-chart").each(function() {
	        			AmetysOrganizationChart._goToMainChart($j(this));
	        		});
	            }
	        }
	        else if (location.hash == '#udroot')
	        {
	        	$j(".organization-chart").each(function() {
        			AmetysOrganizationChart._goToMainChart($j(this));
        		});
	        }
	    });
	            
	    $j(window).bind( 'load', function(e) 
	    {
	    	// Get the firt ul div of all chart
	        $j(".chart-no-focus").each(function() {
	        	var ulDiv = $j(this);

	        	if (location.hash != '')
	        	{
	        		if (location.hash.startsWith("#ud-"))
	        		{
		        		var id = window.location.hash.substring(1);
			            var rootDiv = $j("[data-orgunit-id='" + id + "']", ulDiv);
			            if (rootDiv.length > 0)
		        		{
		        			AmetysOrganizationChart._goToOrgUnitDiv($j(rootDiv));
		        		}
	        		}
	        	}
	        	else
        		{
        			location.hash = "#udroot";
        		}
	        });
	                        
	    });
    }
    
    init();
});
