/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDataProvider;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.external.ExternalizableDataProviderExtensionPoint;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue.Mode;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * Delete orgunit component
 */
public class DeleteOrgUnitComponent extends AbstractDeleteUDContentComponent
{
    /** The avalon role. */
    public static final String ROLE = DeleteOrgUnitComponent.class.getName();
    
    /** Constant for storing the scc identifier into the parameters map */
    public static final String SCC_ID_PARAMETERS_KEY = "sccId";
   
    /** The organisation chart page handler */
    protected OrganisationChartPageHandler _oCPageHandler;
    
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    /** The external data provider extension point */
    protected ExternalizableDataProviderExtensionPoint _externalizableDataProviderExtensionPoint;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _oCPageHandler = (OrganisationChartPageHandler) smanager.lookup(OrganisationChartPageHandler.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
        _externalizableDataProviderExtensionPoint = (ExternalizableDataProviderExtensionPoint) smanager.lookup(ExternalizableDataProviderExtensionPoint.ROLE);
    }
    
    @Override
    public boolean isContentReferenced(Content content, Logger logger)
    {
        List<String> ignoreContentTypes = new ArrayList<>();
        ignoreContentTypes.add(UserDirectoryHelper.ORGUNIT_CONTENT_TYPE);
        ignoreContentTypes.add(UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE);
        
        return _contentHelper.hasReferencingContents(content, ignoreContentTypes, true);
    }
    
    @Override
    protected boolean _checkBeforeDeletion(Content content, Map<String, String> rights, Map<String, Object> results, Logger logger)
    {
        // Check right and lock on content it self
        boolean allRight = _canDeleteContent(content, rights, results);
        
        // Check lock on parent contents
        allRight = _checkParentBeforeDeletion(content, results) && allRight;
        
        // Check lock on users contents
        allRight = _checkUsersBeforeDeletion(content, results) && allRight;
        
        // Check right and lock on children to be deleted or modified
        allRight = _checkChildrenBeforeDeletion(content, rights, results, logger) && allRight;
        
        return allRight;
    }
    
    /**
     * True if the parent content is not locked
     * @param content the content
     * @param results the results map
     * @return true if the parent content is not locked
     */
    protected boolean _checkParentBeforeDeletion(Content content,  Map<String, Object> results)
    {
        boolean allRight = true;
        
        // Check if parents are not locked
        Content parentContent = _oCPageHandler.getParentContent(content);
        if (_isLocked(parentContent))
        {
            @SuppressWarnings("unchecked")
            List<Content> lockedContents = (List<Content>) results.get("locked-contents");
            lockedContents.add(parentContent);
            
            allRight = false;
        }
        
        return allRight;
    }
    
    /**
     * True if the users content is not locked
     * @param content the content
     * @param results the results map
     * @return true if the parent content is not locked
     */
    protected boolean _checkUsersBeforeDeletion(Content content,  Map<String, Object> results)
    {
        boolean allRight = true;
        
        // Check if users are not locked
        for (Content user : _oCPageHandler.getUserContents(content))
        {
            if (_isLocked(user))
            {
                @SuppressWarnings("unchecked")
                List<Content> lockedContents = (List<Content>) results.get("locked-contents");
                lockedContents.add(user);
                
                allRight = false;
            }
        }
        
        return allRight;
    }
    
    /**
     * Browse children to check if deletion could succeed
     * @param contentToCheck The current content to check
     * @param results The result
     * @param rights the map of rights id with its prefix
     * @param logger The logger
     * @return true if the deletion can be processed
     */
    protected boolean _checkChildrenBeforeDeletion(Content contentToCheck, Map<String, String> rights, Map<String, Object> results, Logger logger)
    {
        boolean allRight = true;
        
        List<Content> childOrgUnits = _oCPageHandler.getChildContents(contentToCheck);
        for (Content childOrgUnit : childOrgUnits)
        {
            if (!_canDeleteContent(childOrgUnit, rights, results))
            {
                allRight = false;
            }
            else if (isContentReferenced(childOrgUnit, logger))
            {
                @SuppressWarnings("unchecked")
                List<Content> referencedContents = (List<Content>) results.get("referenced-contents");
                referencedContents.add(childOrgUnit);
                
                allRight = false;
            }
            else
            {
                // Browse children recursively
                allRight = _checkChildrenBeforeDeletion(childOrgUnit, rights, results, logger) && allRight;
            }
        }
        
        return allRight;
    }

    @Override
    protected boolean _removeRelations(Content orgUnit, Map<String, Object> parameters, Logger logger)
    {
        boolean success = true;
        
        // Delete relation to parent and users
        List<Pair<String, Content>> incomingReferences = _contentHelper.getReferencingContents(orgUnit);
        for (Pair<String, Content> reference : incomingReferences)
        {
            Content refContent = reference.getRight();
            String valuePath = reference.getLeft();
            
            try
            {
                // Ignore child orgunits because they are going to be deleted
                if (valuePath.equals(OrganisationChartPageHandler.CHILD_ORGUNIT_ATTRIBUTE_NAME))
                {
                    I18nizableText commentText = new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_WORKFLOW_ACTION_REMOVE_ORGUNIT_REFERENCE_MSG");
                    String comment = _i18nUtils.translate(commentText, refContent.getLanguage());
                    
                    _removeRelation((ModifiableWorkflowAwareContent) refContent, parameters, valuePath, orgUnit, comment);
                }
                else if (valuePath.equals(UserDirectoryHelper.ORGUNITS_ATTRIBUTE))
                {
                    I18nizableText commentText = new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_WORKFLOW_ACTION_REMOVE_USER_REFERENCE_MSG");
                    String comment = _i18nUtils.translate(commentText, refContent.getLanguage());
                    
                    _removeRelation((ModifiableWorkflowAwareContent) refContent, parameters, valuePath, orgUnit, comment);
                }
            }
            catch (WorkflowException | InvalidActionException e)
            {
                logger.error("Unable to remove relation to content \"{}\" ({}) for referencing content \"{}\" ({}) ", orgUnit.getTitle(), orgUnit.getId(), refContent.getTitle(), refContent.getId(), e);
                success = false;
            }
        }
        
        return success;
    }
    
    private void _removeRelation(ModifiableWorkflowAwareContent content, Map<String, Object> parameters, String attribute, Content orgUnit, String comment) throws WorkflowException
    {
        ModelItem modelItem = content.getDefinition(attribute);
        Map<String, Object> externalizableDataContext = new HashMap<>();
        if (parameters.containsKey(SCC_ID_PARAMETERS_KEY))
        {
            externalizableDataContext.put(SynchronizableContentsCollectionDataProvider.SCC_ID_CONTEXT_KEY, parameters.get(SCC_ID_PARAMETERS_KEY));
        }
        
        ExternalizableDataStatus status = _externalizableDataProviderExtensionPoint.isDataExternalizable(content, modelItem, externalizableDataContext) ? ExternalizableDataStatus.EXTERNAL : ExternalizableDataStatus.LOCAL;
        SynchronizableValue value = new SynchronizableValue(List.of(orgUnit.getId()), status);
        value.setMode(Mode.REMOVE);
        
        Map<String, Object> values = Map.of(attribute, value);
        
        Map<String, Object> inputs = new HashMap<>();
        if (StringUtils.isNotEmpty(comment))
        {
            inputs.put("comment", comment);
        }
        
        Map<String, Object> actionParameters = new HashMap<>();
        
        actionParameters.put(EditContentFunction.VALUES_KEY, values);
        // Unlock the content
        actionParameters.put(EditContentFunction.QUIT, true);
        inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, actionParameters);
        // Do not edit the invert relation, we want to keep the relation on the content we are deleting
        inputs.put(EditContentFunction.INVERT_RELATION_ENABLED, false);
    
        _contentWorkflowHelper.doAction(content, _removeReferenceActionId, inputs);
    }

    
    @Override
    protected Set<DeletionInfo> _getContentIdsToDelete(Content content, Map<String, Object> parameters, Map<String, String> rights, boolean hidden, Map<String, Object> results, Logger logger)
    {
        Set<DeletionInfo> toDelete = new HashSet<>();
        
        if (_canDeleteContent(content, rights, results))
        {
            List<String> linkedContents = new ArrayList<>();
            toDelete.add(new DeletionInfo(content.getId(), linkedContents, hidden));
            
            for (Content childOrgUnit : _oCPageHandler.getChildContents(content))
            {
                linkedContents.add(childOrgUnit.getId());
                if (!isContentReferenced(childOrgUnit, logger) && _removeUsersRelation(childOrgUnit, parameters, logger))
                {
                    toDelete.addAll(_getContentIdsToDelete(childOrgUnit, parameters, rights, true, results, logger));
                }
                else
                {
                    // The child program item can not be deleted, remove the relation to the parent and stop iteration
                    @SuppressWarnings("unchecked")
                    List<Content> referencedContents = (List<Content>) results.get("referenced-contents");
                    referencedContents.add(childOrgUnit);
                }
            }
        }
        
        return toDelete;
    }
    
    /**
     * Remove users relation
     * @param orgUnit the orgunit content
     * @param parameters the additional parameters
     * @param logger The logger
     * @return true if relations have been removed
     */
    protected boolean _removeUsersRelation(Content orgUnit, Map<String, Object> parameters, Logger logger)
    {
        boolean success = true;
        
        // Delete relation users
        List<Pair<String, Content>> incomingReferences = _contentHelper.getReferencingContents(orgUnit);
        for (Pair<String, Content> refPair : incomingReferences)
        {
            Content refContent = refPair.getValue();
            String valuePath = refPair.getKey();
            
            try
            {
                // Just remove when path is from users
                if (valuePath.equals(UserDirectoryHelper.ORGUNITS_ATTRIBUTE))
                {
                    I18nizableText commentText = new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_WORKFLOW_ACTION_REMOVE_REFERENCE_MSG");
                    String comment = _i18nUtils.translate(commentText, refContent.getLanguage());
                    
                    _removeRelation((ModifiableWorkflowAwareContent) refContent, parameters, valuePath, orgUnit, comment);
                }
            }
            catch (WorkflowException | InvalidActionException e)
            {
                logger.error("Unable to remove relations to content \"" + orgUnit.getTitle() + " (" + orgUnit.getId() + ")", e);
                success = false;
            }
        }
        
        return success;
    }
}
