/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.synchronize;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.slf4j.Logger;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.plugins.contentio.synchronize.impl.DefaultSynchronizingContentOperator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;

/**
 * Operator to get UDOrgUnit Type content
 */
public class UDOrgUnitTypeOperator extends DefaultSynchronizingContentOperator implements Serviceable
{
    /** The id of orgunit's type content type */
    protected static final String UDORGUNIT_TYPE_CONTENT_TYPE = "org.ametys.plugins.userdirectory.Content.udorgunitType";
    /** The name of the attribute holding the type of orgunit */
    protected static final String UDORGUNIT_ATTRIBUTE_TYPE = "type";
    /** The name of the attribute holding the code of a Orgunit type */
    protected static final String UDORGUNIT_TYPE_ATTRIBUTE_CODE = "code";
    
    /** The Ametys resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public Map<String, List<Object>> transform(ContentType cType, Map<String, List<Object>> remoteValues, Logger logger)
    {
        if (remoteValues.containsKey(UDORGUNIT_ATTRIBUTE_TYPE))
        {
            Map<String, List<Object>> transformedValues = new HashMap<>(remoteValues);
            
            List<Object> contents = new ArrayList<>();
            List<Object> typeCodes = remoteValues.get(UDORGUNIT_ATTRIBUTE_TYPE);
            for (Object typeCode : typeCodes)
            {
                Content udorgUnitTypeContent = _getOrgUnitTypeFromCode(String.valueOf(typeCode));
                if (udorgUnitTypeContent != null)
                {
                    contents.add(udorgUnitTypeContent);
                }
                else
                {
                    logger.warn("The code '{}' does not have a corresponding UDOrgunit Type content. It will be ignored.", typeCode);
                }
            }
            
            transformedValues.put(UDORGUNIT_ATTRIBUTE_TYPE, contents);
            return transformedValues;
        }
        else
        {
            return remoteValues;
        }
    }
    
    /**
     * Get UDOrgUnit Type content from code
     * @param code the code
     * @return The UDOrgUnit Type content or <code>null</code> if not found
     */
    protected Content _getOrgUnitTypeFromCode (String code)
    {
        ContentTypeExpression cTypeExpr = new ContentTypeExpression(Operator.EQ, UDORGUNIT_TYPE_CONTENT_TYPE);
        StringExpression codeExpr = new StringExpression(UDORGUNIT_TYPE_ATTRIBUTE_CODE, Operator.EQ, code);
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(cTypeExpr, codeExpr));
        return _resolver.<Content>query(xpathQuery).stream()
                .findFirst()
                .orElse(null);
    }
}

