/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.transformation.xslt;

import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.page.UserDirectoryPageResolver;
import org.ametys.plugins.userdirectory.page.UserPage;

/**
 * Helper component to be used from XSL stylesheets.
 */
public class UserXSLTHelper extends org.ametys.web.transformation.xslt.AmetysXSLTHelper
{
    /** The resolver for user directory pages */
    protected static UserDirectoryPageResolver _userDirectoryPageResolver;
    /** The user directory helper */
    protected static UserDirectoryHelper _userDirectoryHelper;
    
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _userDirectoryPageResolver = (UserDirectoryPageResolver) manager.lookup(UserDirectoryPageResolver.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) manager.lookup(UserDirectoryHelper.ROLE);
    }
    
    /**
     * Gets the id of the content of the given user
     * @param lang the content language
     * @param userIdentity The user to get content (such as login#population)
     * @return the id of the content of the given user
     * @deprecated Use {@link #getUserContents(String, String)} instead
     */
    @Deprecated
    public static String getUserContent(String lang, String userIdentity)
    {
        if (StringUtils.isBlank(userIdentity))
        {
            return null;
        }
        
        UserIdentity user = UserIdentity.stringToUserIdentity(userIdentity);
        Content userContent = _userDirectoryHelper.getUserContent(user, lang);
        if (userContent != null)
        {
            return userContent.getId();
        }
        
        return null;
    }
    
    /**
     * Gets the id of the content of the given user
     * @param lang the content language
     * @param userIdentity The user to get content (such as login#population)
     * @return the contents of the given user or null if no user
     */
    public static List<Content> getUserContents(String lang, String userIdentity)
    {
        if (StringUtils.isBlank(userIdentity))
        {
            return null;
        }
        
        UserIdentity user = UserIdentity.stringToUserIdentity(userIdentity);
        return _userDirectoryHelper.getUserContents(user, lang);
    }
    
    /**
     * Gets the id of the content of the current connected user
     * @param lang the content language
     * @return the id of the content of the current connected user
     * @deprecated Use {@link #getCurrentUserContents(String)} instead
     */
    @Deprecated
    public static String getCurrentUserContent(String lang)
    {
        UserIdentity currentUser = _currentUserProvider.getUser();
        if (currentUser == null)
        {
            return null;
        }
        
        Content userContent = _userDirectoryHelper.getUserContent(currentUser, lang);
        if (userContent != null)
        {
            return userContent.getId();
        }
        
        return null;
    }
    
    /**
     * Gets the id of the content of the current connected user
     * @param lang the content language
     * @return the contents of the current connected user
     */
    public static List<Content> getCurrentUserContents(String lang)
    {
        UserIdentity currentUser = _currentUserProvider.getUser();
        if (currentUser == null)
        {
            return null;
        }
        
        return _userDirectoryHelper.getUserContents(currentUser, lang);
    }
    
    /**
     * Gets the id of the page of the current user
     * @param lang The language
     * @return the id of the page of the user content or null if not found
     */
    public static String getCurrentUserPage(String lang)
    {
        UserIdentity currentUser = _currentUserProvider.getUser();
        if (currentUser == null)
        {
            return null;
        }
        
        String siteName = site();
        List<Content> userContents = _userDirectoryHelper.getUserContents(currentUser, lang);
        for (Content content : userContents)
        {
            String userPage = _getUserPage(content, siteName);
            if (userPage != null)
            {
                return userPage;
            }
        }
        
        return null;
    }
    
    
    /**
     * Gets the id of the page of the user content
     * @param contentId The user content id
     * @return the id of the page of the user content
     */
    public static String getUserPage(String contentId)
    {
        String siteName = site();
        return getUserPage(contentId, siteName);
    }
    
    /**
     * Gets the id of the page of the user content
     * @param contentId The user content id
     * @param siteName the site name
     * @return the id of the page of the user content
     */
    public static String getUserPage(String contentId, String siteName)
    {
        if (contentId == null)
        {
            return null;
        }
        
        try
        {
            Content content = _ametysObjectResolver.resolveById(contentId);
            return _getUserPage(content, siteName);
        }
        catch (UnknownAmetysObjectException e)
        {
            _logger.warn("Can't find user content for id '" + contentId + "'", e);
            return null;
        }
    }
    
    private static String _getUserPage(Content content, String siteName)
    {
        UserPage userPage = _userDirectoryPageResolver.getUserPage(content, StringUtils.isBlank(siteName) ? site() : siteName, lang());
        return userPage != null ? userPage.getId() : null;
    }
}
