/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.userdataprovider;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.data.File;
import org.ametys.cms.repository.Content;
import org.ametys.core.user.User;
import org.ametys.core.user.dataprovider.UserDataProvider;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebHelper;

/**
 * A {@link UserDataProvider} that finds data from the ContentUser associated to the User or UserIdentity given
 */
public class ContentUserDataProvider extends AbstractLogEnabled implements UserDataProvider, Serviceable, Initializable, Contextualizable
{
    /** Attribute path for user's image */
    public static final String USER_CONTENT_IMAGE_PATH = "illustration/image";
    private static final Comparator<Content> __COMPARATOR = Comparator.comparing(Content::getName);
    private Set<String> _supportedData;
    private UserDirectoryHelper _userDirectoryHelper;
    private ContentTypesHelper _contentTypesHelper;
    private Context _context;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _userDirectoryHelper = (UserDirectoryHelper) manager.lookup(UserDirectoryHelper.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public int getPriority()
    {
        return 10000;
    }

    public void initialize() throws Exception
    {
        _supportedData = new HashSet<>();
        
        // Get the content types of a ud user
        Map<String, Object> contentTypesList = _contentTypesHelper.getContentTypesList(Collections.singletonList("org.ametys.plugins.userdirectory.Content.user"),
                true /* inherited */,
                false /* checkRights */,
                true /* includePrivate */,
                false /* includeMixins */,
                false /* includeAbstract*/);
       
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> contentTypes = (List<Map<String, Object>>) contentTypesList.get("contentTypes");
        
        // Get the model items names of all the model items that can be in a ud user
        for (Map<String, Object> contentType : contentTypes)
        {
            Collection< ? extends ModelItem> modelItems = _contentTypesHelper.getModelItems(List.of(contentType.get("id"))
                                                                             .toArray(String[]::new));
            
            _supportedData.addAll(modelItems.stream()
                   .map(ModelItem::getName)
                   .collect(Collectors.toSet()));
        }
    }
    
    public boolean supports(String element)
    {
        // Check if the element requested can be supported. Can return true but not have a value for a specific user.
        return User.IMAGE_DATA_ID.equals(element) || _supportedData.contains(element);
    }
    
    @Override
    public boolean hasValue(User user, String dataId)
    {
        // Sort users by name in order to always retrieve the same value by dataId
        List<Content> contentUsers = _getSortedContentUsers(user);
        
        String dataIdToCheck = User.IMAGE_DATA_ID.equals(dataId) ? USER_CONTENT_IMAGE_PATH : dataId;
        
        // Get the ud users and check if it has a value for the data requested
        for (Content contentUser : contentUsers)
        {
            if (hasValue(contentUser, dataIdToCheck))
            {
                return true;
            }
        }
        
        return false;
    }
    
    private boolean hasValue(Content contentUser , String dataId)
    {
        return contentUser != null && contentUser.hasValue(dataId);
    }

    public Object getValue(User user, String dataId)
    {
        // Sort users by name in order to always retrieve the same value by dataId
        List<Content> contentUsers = _getSortedContentUsers(user);
        
        String dataIdToCheck = User.IMAGE_DATA_ID.equals(dataId) ? USER_CONTENT_IMAGE_PATH : dataId;

        // Get the ud users and retrieve the value wanted if it exists
        for (Content contentUser : contentUsers)
        {
            if (hasValue(contentUser, dataIdToCheck))
            {
                Object value = contentUser.getValue(dataIdToCheck);
                
                if (USER_CONTENT_IMAGE_PATH.equals(dataIdToCheck) && value != null && value instanceof File file)
                {
                    return new UserDirectoryImageAccessor(file, getLogger());
                }
                else
                {
                    return value;
                }
            }
        }
        
        return null;
    }
    
    private List<Content> _getSortedContentUsers(User user)
    {
        Request request = ContextHelper.getRequest(_context);
        String lang = WebHelper.findLanguage(request);
        
        List<Content> userContents = new ArrayList<>(_userDirectoryHelper.getUserContents(user.getIdentity(), lang));
        userContents.sort(__COMPARATOR);
        return userContents;
    }
}
