/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webanalytics.ga;

import java.net.URISyntaxException;
import java.util.Random;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.util.URIUtils;
import org.ametys.web.analytics.AbstractWebAnalyticsProvider;
import org.ametys.web.repository.site.Site;

/**
 * The google analytics provider
 */
public class GoogleAnalyticsProvider extends AbstractWebAnalyticsProvider
{
    /** The analytics GIF URL. */
    public static final String GA_GIF_URL = "https://www.google-analytics.com/__utm.gif";
    
    /** The analytics API version. */
    public static final String GA_VERSION = "5.3.8";
    
    /** The domain hash. */
    public static final String DOMAIN_HASH = "109627128"; // TODO Really hash the domain name?
    
    private static final Random _RNG = new Random((long) Math.random() * System.nanoTime());
    
    public String getEventImageUri(Site site, String category, String action, String label, int value, boolean usePlaceholderTokens)
    {
        String gaWebId = site.getValue("google-web-property-id");
        if (StringUtils.isNotBlank(gaWebId))
        {
            String identifier = _getEventIdentifier(category, action, label, value);
            return _getEventImageUri(gaWebId, identifier, usePlaceholderTokens);
        }
        
        return null;
    }
    
    /**
     * Get the event identifier
     * @param category the event category
     * @param action the event action
     * @param label the event label
     * @param value the event value
     * @return the event identifier
     */
    protected String _getEventIdentifier(String category, String action, String label, int value)
    {
        try
        {
            StringBuilder eventId = new StringBuilder();
            
            eventId.append("5(");
            eventId.append(_encodeValue(category));
            eventId.append('*').append(_encodeValue(action));
            eventId.append('*').append(_encodeValue(label));
            eventId.append(')');
            if (value > 0)
            {
                eventId.append('(').append(value).append(')');
            }
            return eventId.toString();
        }
        catch (URISyntaxException e)
        {
            getLogger().error("An error occurred computing the event gif uri", e);
        }
        
        return null;
    }
    
    /**
     * Get an event image URI.
     * @param gaWebId the GA web ID.
     * @param eventIdentifier the event identifier.
     * @param usePlaceholderTokens True to use tokens instead of randomly generated values
     * @return the event image URI.
     */
    protected String _getEventImageUri(String gaWebId, String eventIdentifier, boolean usePlaceholderTokens)
    {
        long now = System.currentTimeMillis() / 1000L;
        
        StringBuilder cookieValue = new StringBuilder();
        // UTMA
        cookieValue.append("__utma=").append(DOMAIN_HASH).append('.').append(usePlaceholderTokens ? RANDOM_NUMBER_TOKEN : _RNG.nextInt(Integer.MAX_VALUE));
        cookieValue.append('.').append(now).append('.').append(now).append('.').append(now).append(".1");
        // UTMZ
        cookieValue.append(";+__utmz=").append(DOMAIN_HASH).append('.').append(now);
        cookieValue.append(".1.1.utmcsr=(direct)|utmccn=(direct)|utmcmd=(none);");
        
        StringBuilder uri = new StringBuilder();
        
        uri.append(GA_GIF_URL);
        uri.append("?utmwv=").append(GA_VERSION);
        uri.append("&utmn=").append(usePlaceholderTokens ? RANDOM_NUMBER_TOKEN : _RNG.nextInt(Integer.MAX_VALUE));
        uri.append("&utmt=event");
        uri.append("&utme=").append(eventIdentifier);
        uri.append("&utmcs=UTF-8");
        uri.append("&utmje=1");
        uri.append("&utmhid=").append(usePlaceholderTokens ? RANDOM_NUMBER_TOKEN : _RNG.nextInt(Integer.MAX_VALUE));
        // TODO utmhn with main domain?
        uri.append("&utmr=-");
        uri.append("&utmac=").append(gaWebId);
        
        // Cookie
        String encodedCookie = URIUtils.encodeParameter(cookieValue.toString());
        uri.append("&utmcc=").append(encodedCookie);
        
        return uri.toString();
    }
    
    public String getEventLinkCampaignParams(Site site, String campaign, String medium, String source)
    {
        StringBuilder linkCampaignParams = new StringBuilder();
        linkCampaignParams.append("utm_campaign=");
        linkCampaignParams.append(campaign);
        linkCampaignParams.append("&utm_medium=");
        linkCampaignParams.append(medium);
        linkCampaignParams.append("&utm_source=");
        linkCampaignParams.append(source);
        
        return linkCampaignParams.toString();
    }
    
    public void sendTrackingInformation(Site site) throws Exception
    {
        // Do nothing for Google Analytics
    }
}
