/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

// Edit catalog site to assign new site type
// Edit project site to assign edited site type

const CatalogSiteType = Java.type("org.ametys.plugins.workspaces.catalog.CatalogSiteType");
const ProjectSiteType = Java.type("org.ametys.plugins.workspaces.project.ProjectWorkspaceSiteType");

function migrateProjetsSiteType(s) {
    const xpath = "//element(*, ametys:site)[@ametys:type = 'org.ametys.plugins.workspaces.project.sitetype']";
    const projectSites = Repository.resolver.query(xpath, s);

    var total = projectSites.getSize();
    logger.info(`migration ${total} project site to new site type`);
    projectSites.forEach(function(projectSite) {
        projectSite.setType(ProjectSiteType.TYPE_ID);
    });

    s.save();
    logger.info(`project sites were successfully migrated`)
}

function migrateCatalogSiteType(catalogSiteName, s) {
    const xpath = `//element(${catalogSiteName}, ametys:site)`;
    const catalogSites = Repository.resolver.query(xpath, s);

    logger.info(`migrating catalog site ${catalogSiteName} to new site type and parameters`)
    catalogSites.forEach(function(catalogSite) {
        
        catalogSite.setType(CatalogSiteType.TYPE_ID);
        
        let query = "//element(" + catalogSiteName + ", ametys:site)//element(*, ametys:zoneItem)[ametys-internal:service = 'org.ametys.plugins.workspaces.service.ProjectsCatalogue']";
        let catalogServices = Repository.resolver.query(query, s);
        
        catalogServices.forEach(function(zoneItem) {
            var serviceParamNode = zoneItem.getServiceParameters().getRepositoryData().getNode(); // can't use the model directly as the parameters doesn't exist anymore
            
            if (serviceParamNode.hasProperty("ametys:urlPrefix"))
            {
	            var urlPrefix = serviceParamNode.getProperty("ametys:urlPrefix");
                catalogSite.setValue(CatalogSiteType.PROJECT_URL_PREFIX_SITE_PARAM, urlPrefix.getString());
                urlPrefix.remove();
            }
            if (serviceParamNode.hasProperty("ametys:skin"))
            {
	            var skin = serviceParamNode.getProperty("ametys:skin");
	            catalogSite.setValue(CatalogSiteType.PROJECT_DEFAULT_SKIN_SITE_PARAM, skin.getString());
	            skin.remove();
            }
        });
    });

    s.save();
    logger.info(`catalog site sucessfully migrated`);
}

// First migrate projet
migrateProjetsSiteType(Repository.session);
migrateProjetsSiteType(Repository.liveSession);

// Then catalog
let catalogSiteName = org.ametys.runtime.config.Config.getInstance().getValue('workspaces.catalog.site.name');
if (catalogSiteName == null)
{
    throw new org.ametys.core.migration.MigrationException(`Catalog site name is not defined in configuration. Migration is impossible.`);
}
migrateCatalogSiteType(catalogSiteName, Repository.session);
migrateCatalogSiteType(catalogSiteName, Repository.liveSession);
