/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';
import { useCalendarsStore } from '@/stores/calendars'
import { useEventFiltersStore } from '@/stores/eventFilters'
import { useResourcesStore } from '@/stores/resources'
import { useUserRightsStore } from '@/stores/userRights';
import { useTagsStore } from '@common/store/project-tags/tags'

/**
 * Pinia store for handling events of a calendar or resources
 */
export const useEventsStore = defineStore('events', {
    state: () => (
    {
        events: [], // list of events
        asideEvents: [], // list of the next events for the aside panel
    }),
    getters: {
        
        /**
         * Get the list of filtered events with their colors and occurrences
         */
        getEvents: state => {

            const calendarsStore = useCalendarsStore();
            const type  = calendarsStore.type;
            
            const filtersStore = useEventFiltersStore();
            const filterByResourceCalendar = filtersStore.filterByResourceCalendar
            const filterByCalendar = filtersStore.filterByCalendar

            let events = state.events;
            events = events.filter((event) => !event.calendar.isTaskCalendar || !event.calendar.isTaskCalendarDisabled);
            
            if (type == 'calendar' && !filterByResourceCalendar) events = events.filter((event) => event.calendarId != state.$resourceCalendar.id)

            if (type == 'resources') events = events.filter((event) => event.hasOwnProperty('resourceIds') && event.resourceIds.length > 0)

            if (filterByResourceCalendar) events = events.filter((event) => event.calendarId == state.$resourceCalendar.id)

            if (typeof filterByCalendar == 'object' && filterByCalendar.id) events = events.filter((e) => (e.calendarId == filterByCalendar.id))

            const data = [];
            const resourceStore = useResourcesStore();
            const userRightsStore = useUserRightsStore();
            var resourceList = resourceStore.resources.map(function(resource)
            {
                return resource['id'];
            });
            events.forEach((event) =>
            {
                const isResource = event.calendarId == state.$resourceCalendar.id
                const color = state.$colorsMap[event.calendar.color] || state.$colorsMap["col1"];
                var filteredResources = event.resourceIds
                    .filter((resource) => resourceList.includes(resource));
                event.occurrences.forEach((occurrence) =>
                {
                    data.push({
                        ...event,
                        color: color.bg,
                        start: occurrence.startDate,
                        end: event.fullDay ? occurrence.endDateNextDay : occurrence.endDate,
                        endDateNextDay: occurrence.endDateNextDay,
                        allDay: event.fullDay,
                        editable: event.isModifiable && userRightsStore.userRights.canEditEvent && (!isResource || userRightsStore.userRights.canHandleResource),
                        resourceEditable: event.isModifiable && userRightsStore.userRights.canEditEvent && (!isResource || userRightsStore.userRights.canHandleResource),
                        id: event.id + occurrence.occurrenceDate,
                        extendedProps: {
                            ...event,
                            eventId: event.id,
                            color: color.bg,
                            occurrenceDate: occurrence.occurrenceDate,
                            occurrenceEndDate: occurrence.endDate,
                            resourceIds: filteredResources,
                            originalStart: event.startDate,
                            originalOccurrenceStart: occurrence.startDate,
                            isResource: isResource,
                        }
                    });
                });
            });
            return data;
        },

        /**
         * Get the list of filtered aside events with their colors and occurrences
         */
        getAsideEvents: state => {
            const calendarsStore = useCalendarsStore();
            const type  = calendarsStore.type;
            
            const filtersStore = useEventFiltersStore();
            const filterByResourceCalendar = filtersStore.filterByResourceCalendar
            const filterByCalendar = filtersStore.filterByCalendar

            let events = state.asideEvents;
            events = events.filter((event) => !event.calendar.isTaskCalendar || !event.calendar.isTaskCalendarDisabled);

            if (type == 'calendar' && !filterByResourceCalendar) events = events.filter((event) => event.calendarId != state.$resourceCalendar.id)

            if (type == 'resources') events = events.filter((event) => event.hasOwnProperty('resourceIds') && event.resourceIds.length > 0)

            if (filterByResourceCalendar) events = events.filter((event) => event.calendarId == state.$resourceCalendar.id)

            if (typeof filterByCalendar == 'object' && filterByCalendar.id) events = events.filter((e) => (e.calendarId == filterByCalendar.id))

            var data = [];

            events.forEach((event) => {
                const color = state.$colorsMap[event.calendar.color] || state.$colorsMap["col1"];

                event.occurrences.forEach((occurrence) => {
                    data.push({
                        ...event,
                        color: color.bg,
                        start: occurrence.startDate,
                        end: occurrence.endDate,
                        allDay: event.fullDay,
                        id: event.id + occurrence.occurrenceDate,
                        extendedProps: {...event,
                            color: color.bg
                        }
                    });
                });
            });

            data = data.filter((occurrence) => new Date(occurrence.end).getTime() >= new Date().getTime())
                    .sort((a, b) => new Date(a.start).getTime() - new Date(b.start).getTime())
                    .slice(0, 4);

            return data;
        },
    },
    actions:
    {
        /**
         * Load the events for the current project
         * @param {Date} start the start date of the range to load events
         * @param {Date} end the end date of the range to load events
         */
        async loadEvents(start, end)
        {
            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.events.CalendarEventDAO',
                methodName: 'getEvents',
                parameters: [start, end],
            })
            .then((events) => {
                this.events = events;
                AmetysFront.Event.fire('events-loaded', {events});
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_LOAD_EVENT_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                });
            });
        },

         /**
          * Load the next events for the aside panel
          * @param {Date} start the start date of the range to load events
          * @param {Date} end the end date of the range to load events
          */
        async loadAsideEvents(start, end )
        {
            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.events.CalendarEventDAO',
                methodName: 'getEvents',
                parameters: [start, end],
            })
            .then((events) => {
                this.asideEvents = events;
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_LOAD_EVENT_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                });
            });
        },

        /**
         * Update an event
         * @param {Object} updatedEvent the event to update
         * @param {Date} calendarViewStartDate the start date of the calendar view
         * @param {Date} calendarViewEndDate the end date of the calendar view
         */
        async updateEvent({updatedEvent, calendarViewStartDate, calendarViewEndDate })
        {

            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_EDIT_EVENT_BEGIN_MSG
            });

            // Add edition action id
            updatedEvent.actionId = 2;

            return await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.events.CalendarEventDAO',
                methodName: 'editEvent',
                parameters: [updatedEvent, calendarViewStartDate, calendarViewEndDate],
            })
            .then((data) => {
                const index = _getEventIndexById(this.events, updatedEvent.id);
                this.events.splice(index, 1, data.oldEventData);
                const asideIndex = _getEventIndexById(this.asideEvents, updatedEvent.id);
                this.asideEvents.splice(asideIndex, 1, data.oldEventData);
                if (updatedEvent.id != data.id)
                {
                    this.events.push(data.newEventData);
                    this.asideEvents.push(data.newEventData);
                }

                const tagsStore = useTagsStore();
                tagsStore.addTags(data.newTags);

                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_EDIT_EVENT_END_MSG
                });
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_EDIT_EVENT_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                });
            });
        },
         
        /**
         * Delete an event
         * @param {Object} payload the payload containing the event id, occurrence and choice
         * @param {String} payload.id the id of the event to delete
         * @param {String} payload.occurrence the occurrence of the event to delete (if applicable)
         * @param {String} payload.choice the choice of deletion, either "unit" for a single occurrence or "all" for all occurrences
         */
        async deleteEvent(payload)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DELETE_EVENT_BEGIN_MSG
            });

            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.events.CalendarEventDAO',
                methodName: 'deleteEvent',
                parameters: [payload.id, payload.occurrence, payload.choice],
            })
            .then(() =>
            {
                const index = _getEventIndexById(this.events, payload.id);
                const asideIndex = _getEventIndexById(this.asideEvents, payload.id);
                if (payload.choice == "unit")
                {
                    const event = this.events[index];
                    const occurenceIndex = _getEventOccurrenceIndexById(event.occurrences, payload.id + "$" + payload.occurrence);
                    event.occurrences.splice(occurenceIndex, 1);
                    const asideEvent = this.asideEvents[asideIndex];
                    const asideOccurenceIndex = _getEventOccurrenceIndexById(event.occurrences, payload.id + "$" + payload.occurrence);
                    if (asideEvent != undefined && asideOccurenceIndex != -1)
                    {
                        asideEvent.occurrences.splice(asideOccurenceIndex, 1);
                    }
                }
                else
                {
                    this.events.splice(index, 1);
                    this.asideEvents.splice(asideIndex, 1);
                }
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DELETE_EVENT_END_MSG
                });
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DELETE_EVENT_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                });
            });
        },
         
        /**
         * Add an event
         * @param {Object} addedEvent the event to add
         * @param {Date} calendarViewStartDate the start date of the calendar view
         * @param {Date} calendarViewEndDate the end date of the calendar view
         */
        async addEvent({addedEvent, calendarViewStartDate, calendarViewEndDate })
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_EVENT_BEGIN_MSG
            });

           // Add creation action id
           addedEvent.actionId = 1;

           return await callMethod({
               role: 'org.ametys.plugins.workspaces.calendars.events.CalendarEventDAO',
               methodName: 'addEvent',
               parameters: [addedEvent, calendarViewStartDate, calendarViewEndDate],
           })
           .then( data  =>
           {

               this.events.push(data.eventDataWithFilteredOccurences);
               // Deep clone object to handle possible future operations such as splice
               this.asideEvents.push(JSON.parse(JSON.stringify(data.eventDataWithFilteredOccurences)));
               
               const tagsStore = useTagsStore();
               tagsStore.addTags(data.newTags);
               AmetysFront.Event.fire('loaderEnd', {
                   text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_CREATE_EVENT_END_MSG
               });
           })
           .catch((error) =>
           {
               AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_EVENT_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
               });
           });
        },

        /**
         * Update the calendar of events
         * @param {Object} calendar the calendar object to update
         */
        async updateEventCalendar(calendar)
        {
            this.events.forEach(event =>
            {
                if(event.calendar && event.calendar.id == calendar.id)
                {
                    event.calendar = calendar;
                }
            });
            
            this.asideEvents.forEach(event => 
            {
                if(event.calendar && event.calendar.id == calendar.id)
                {
                    event.calendar = calendar;
                }
            });
        },
        
        /**
         * Update the calendar of events
         * @param {Object} calendar the calendar object to update
         */
        async removeCalendarFromEvents(calendarId)
        {
            this.events = this.events.filter(event => {
                return event.calendar.id !== calendarId
            });
            this.asideEvents = this.asideEvents.filter(event => {
                return event.calendar.id !== calendarId
            });
        },

    }

})

/**
 * Get the index of the event by the id
 * @param {Object[]} events the list of events to search into
 * @param {String} id the id of the event to find
 */
function _getEventIndexById(events, id)
{
    return events.findIndex((event) => event.id.toString() === id.toString());
}

/**
 * Get the index of the occurrence by the id
 * @param {Object[]} occurrences the list of occurrences to search into
 * @param {String} id the id of the occurrence to find
 */
function _getEventOccurrenceIndexById(occurrences, id)
{
    return occurrences.findIndex((occurrence) => occurrence.id.toString() === id.toString());
}
