/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';

/**
 * Pinia store for handling resources
 */
export const useResourcesStore = defineStore('resources', {
    state: () => (
    {
        resources: [], // list of resources
        resourcesAvailability: [], // list of resources with their availability
    }),
    actions:
    {
        /**
         * Load the resources from the server
         */
        async loadResources()
        {
            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.resources.CalendarResourceDAO',
                methodName: 'getResources',
            })
            .then((data ) => {
                data.sort((a, b) => a.title.localeCompare(b.title))
                this.resources = data
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_LOAD_RESOURCE_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                })
            })
        },

        /**
         * Load the resources availability for a given date range and event
         * @param {Object} params the parameters for loading resources availability
         * @param {string} params.id the id of the resource
         * @param {Date} params.startDate the start date of the range
         * @param {Date} params.endDate the end date of the range
         * @param {string} params.eventStartDateAsStr the start date of the event as a string
         * @param {string} params.eventEndDateAsStr the end date of the event as a string
         * @param {string} params.recurrenceType the recurrence type of the event
         * @param {boolean} params.allDay - Whether the event is all day
         * @param {string} params.originalOccurrenceStartAsStr the start date of the original occurrence as a string
         * @param {string} params.zone the time zone of the event
         */
        async loadResourcesWithAvailability({ id, startDate, endDate, eventStartDateAsStr, eventEndDateAsStr, recurrenceType, allDay, originalOccurrenceStartAsStr, zone })
        {
            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.resources.CalendarResourceDAO',
                methodName: 'loadResourcesWithAvailability',
                parameters: [id, startDate, endDate, eventStartDateAsStr, eventEndDateAsStr, recurrenceType, allDay, originalOccurrenceStartAsStr, zone],
            })
            .then((data) => {
                this.resourcesAvailability = data
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_LOAD_RESOURCE_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                })
            })
        },

        /**
         * Add a new resource
         * @param {Object} resource the resource object to add
         * @param {String} resource.title the title of the resource
         * @param {String} resource.icon the icon of the resource
         * @param {String} resource.instructions the instructions of the resource
         */
        async addResource(resource)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_RESOURCE_BEGIN_MSG
            });
            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.resources.CalendarResourceDAO',
                methodName: 'addResource',
                parameters: [
                    resource.title,
                    resource.icon,
                    resource.instructions,
                    false
                ],
            })
            .then((data) => {

                this.resources.push(data)
                this.resources.sort((a, b) => a.title.localeCompare(b.title))
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_RESOURCE_END_MSG
                });
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_RESOURCE_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                })
            })
        },
        
        /**
         * Update an existing resource
         * @param {Object} resource the resource to update
         * @param {String} resource.id the id of the resource
         * @param {String} resource.title the title of the resource
         * @param {String} resource.icon the icon of the resource
         * @param {String} resource.instructions the instructions of the resource
         */
        async updateResource(resource)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_EDIT_RESOURCE_BEGIN_MSG
            });

            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.resources.CalendarResourceDAO',
                methodName: 'editResource',
                parameters: [
                    resource.id,
                    resource.title,
                    resource.icon,
                    resource.instructions
                ],
            })
            .then((data) => {
                const index = this.resources.findIndex((i) => i.id == data.id)
                if (index != -1) this.resources.splice(index, 1, data)
                this.resources.sort((a, b) => a.title.localeCompare(b.title))
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_EDIT_RESOURCE_END_MSG
                });
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_EDIT_RESOURCE_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                })
            })
        },
        
        /*
         * Delete a resource
         * @param {String} id the id of the resource to delete
         */
        async deleteResource(id)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DELETE_RESOURCE_BEGIN_MSG
            });

            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.resources.CalendarResourceDAO',
                methodName: 'deleteResource',
                parameters: [id],
            })
            .then((/*{ data }*/) => {
                const index = this.resources.findIndex((i) => i.id == id)
                if (index != -1)
                {
                    this.resources.splice(index, 1)
                }
                AmetysFront.Event.fire('loaderEnd', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DELETE_RESOURCE_END_MSG,
                });
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DELETE_RESOURCE_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                })
            })
        },
    }
})

