/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import { defineStore } from 'pinia'
import { useFileActivityStore } from '@/stores/fileActivity'
import { useCommentStore } from '@/stores/comment'

/**
 * Pinia store for managing the preview state of an item.
 */
export const usePreviewItemStore = defineStore('previewItem', {
    state: () => ({
        info: false, // Whether the info panel is open
        item: {} // The currently previewed item
    }),
    getters: {
        /**
         * Returns the item id if it's not a folder, otherwise false.
         */
        itemId: (state) => {
            if (state.item && state.item.type !== 'folder') {
                return state.item.id;
            } else {
                return false;
            }
        }
    },
    actions: {
        
        /**
         * Sets the current item and triggers related actions.
         * @param {Object} payload rhe payload
         * @param {Object} payload.item The item to preview
         * @param {boolean} payload.notOpen Whether to not open the info panel
         * @param {boolean} payload.commenter Whether the preview should open on comment tab
         */
        setItem(payload) {
            this.item = payload.item;

            if (payload.item.type === 'file') {
                // These would need to be implemented as separate stores or composables
                const fileActivityStore = useFileActivityStore()
                const commentsStore = useCommentStore()
                fileActivityStore.setLoadBy(4);
                fileActivityStore.loadActivities(payload.item);
                commentsStore.loadComments(payload.item);

                if (payload.commenter) {
                    AmetysFront.Event.fire('commenter');
                } else {
                    AmetysFront.Event.fire('preview');
                }
            }
            if (!payload.notOpen) {
                this.info = true;
            }
        },
        /**
         * Sets the info state.
         * @param {boolean} info
         */
        setInfo(info) {
            this.info = info;
        }
    }
});