/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';
import { useTasksStore } from '@/stores/tasks'


/**
 * Pinia store for comments on tasks
 */
export const useCommentStore = defineStore('comment', {
    actions:
    {
        /**
         * Add a comment to a task
         * @param {String} ametysObjectId the Ametys object id of the task
        * @param {string} text the text of the comment as it is stored
         */
        async addComment(ametysObjectId, text)
        {
            const tasksStore = useTasksStore();
            return new Promise(function(resolve, reject)
            {
                AmetysFront.Event.fire('loaderStart', {
                  text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_BEGIN_COMMENT_ADD
                });

                callMethod({
                  role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                  methodName: 'commentTask',
                  parameters: [ametysObjectId, text]
                })
                .then(task => {
                    tasksStore.transformTask(task);
                    tasksStore.updateTaskInState(task)

                    resolve({comments: task.comments});

                    AmetysFront.Event.fire('loaderEnd', {
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_END_COMMENT_ADD
                    });
                })
                .catch(error => {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_COMMENT_TASK_ERROR_MSG,
                        text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                    reject();
                })
            });
        },

        /**
         * Edit a comment of a task
         * @param {String} ametysObjectId the Ametys object id of the task
         * @param {String} commentId the id of the comment to edit
         * @param {String} text the new text of the comment
         */
        async editComment(ametysObjectId, commentId, text)
        {
            const tasksStore = useTasksStore();
            return new Promise(function(resolve, reject)
            {
                AmetysFront.Event.fire('loaderStart', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_BEGIN_COMMENT_EDITED
                });

                callMethod({
                    role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                    methodName: 'editCommentTask',
                    parameters: [ametysObjectId, commentId, text]
                })
                .then(task => {
                    tasksStore.transformTask(task);
                    tasksStore.updateTaskInState(task)

                    resolve({comments: task.comments});

                    AmetysFront.Event.fire('loaderEnd', {
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_END_COMMENT_EDITED
                    });
                })
                .catch(error => {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_EDIT_COMMENT_TASK_ERROR_MSG,
                        text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                    reject();
                })
            });
        },

        /**
         * Add a sub-comment to a comment of a task
         * @param {String} ametysObjectId the Ametys object id of the task
         * @param {String} commentId the id of the comment to which the sub-comment is added
         * @param {String} text the text of the sub-comment
         */
        async addSubComment(ametysObjectId, commentId, text)
        {
            const tasksStore = useTasksStore();
            return new Promise(function(resolve, reject)
            {
                AmetysFront.Event.fire('loaderStart', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_BEGIN_COMMENT_ADD
                });

                callMethod({
                    role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                    methodName: 'answerCommentTask',
                    parameters: [ametysObjectId, commentId, text]
                })
                .then(task => {
                    tasksStore.transformTask(task);
                    tasksStore.updateTaskInState(task)

                    resolve({comments: task.comments});

                    AmetysFront.Event.fire('loaderEnd', {
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_END_COMMENT_ADD
                    });
                })
                .catch(error => {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_COMMENT_TASK_ERROR_MSG,
                        text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                    reject();
                })
            });
        },

        /**
         * Like or unlike a comment of a task
         * @param {String} ametysObjectId the Ametys object id of the task
         * @param {String} commentId the id of the comment to like or unlike
         * @param {Boolean} liked true to like the comment, false to unlike it
         */
        async likeOrUnlikeComment(ametysObjectId, commentId, liked)
        {
            const tasksStore = useTasksStore();
            return new Promise(function(resolve, reject)
            {
                AmetysFront.Event.fire('loaderStart', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_BEGIN_COMMENT_LIKE
                });

                callMethod({
                    role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                    methodName: 'likeOrUnlikeCommentTask',
                    parameters: [ametysObjectId, commentId, liked]
                })
                .then(task => {
                    tasksStore.transformTask(task);
                    tasksStore.updateTaskInState(task)

                    resolve({comments: task.comments});

                    AmetysFront.Event.fire('loaderEnd', {
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_END_COMMENT_LIKE
                    });
                })
                .catch(error => {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_REACT_COMMENT_TASK_ERROR_MSG,
                        text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                    reject();
                })
            });
        },

        /**
         * Delete a comment of a task
         * @param {String} ametysObjectId the Ametys object id of the task
         * @param {String} commentId the id of the comment to delete
         */
        async deleteComment(ametysObjectId, commentId)
        {
            const tasksStore = useTasksStore();
            return new Promise(function(resolve, reject)
            {
                AmetysFront.Event.fire('loaderStart', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_BEGIN_COMMENT_DELETE
                });

                callMethod({
                    role: 'org.ametys.plugins.workspaces.tasks.WorkspaceTaskDAO',
                    methodName: 'deleteCommentTask',
                    parameters: [ametysObjectId, commentId]
                })
                .then(task => {
                    tasksStore.transformTask(task);
                    tasksStore.updateTaskInState(task)

                    resolve({comments: task.comments});

                    AmetysFront.Event.fire('loaderEnd', {
                        text: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_ADD_TASK_LOADING_END_COMMENT_DELETE
                    });
                })
                .catch(error => {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_DELETE_COMMENT_TASK_ERROR_MSG,
                        text : i18n.PLUGINS_WORKSPACES_PROJECT_MODULE_TASKS_GENERAL_ERROR_TEXT,
                        details: error
                    });
                    reject();
                })
            });
        },
    }
})
