/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.content.indexing.solr.SolrResourceGroupedMimeTypes;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.tag.TaggableAmetysObject;
import org.ametys.plugins.workspaces.project.ProjectConstants;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tags.ProjectTagsDAO;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.URIPrefixHandler;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Utility classes for workspaces
 *
 */
public class WorkspacesHelper extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** Avalon role */
    public static final String ROLE = WorkspacesHelper.class.getName();

    /** The right manager */
    protected RightManager _rightManager;
    
    /** The workspace module extension point */
    protected WorkspaceModuleExtensionPoint _workspaceModuleEP;

    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    
    /**
     * Enumeration for file type
     */
    public enum FileType
    {
        /** Text document */
        TEXT,
        /** PDF document */
        PDF,
        /** Spreadsheet */
        SPREADSHEET,
        /** Presentation */
        PRES,
        /** Image */
        IMAGE,
        /** Video */
        VIDEO,
        /** Audio */
        AUDIO,
        /** Archive */
        ARCHIVE,
        /** Unknown type */
        UNKNOWN
    }

    private Context _context;
    private org.apache.cocoon.environment.Context _cocoonContext;
    private URIPrefixHandler _prefixHandler;
    private ProjectTagsDAO _projectTagsDAO;
    private SiteManager _siteManager;
    private ProjectManager _projectManager;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (org.apache.cocoon.environment.Context) _context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _prefixHandler = (URIPrefixHandler) smanager.lookup(URIPrefixHandler.ROLE);
        _projectTagsDAO = (ProjectTagsDAO) smanager.lookup(ProjectTagsDAO.ROLE);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);

        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
        _workspaceModuleEP = (WorkspaceModuleExtensionPoint) smanager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Determines if the resource file is an image
     * @param file the resource
     * @return true if the resource file is an image
     */
    public boolean isImage(Resource file)
    {
        return SolrResourceGroupedMimeTypes.getGroup(file.getMimeType())
            .map(groupMimeType -> FileType.IMAGE == FileType.valueOf(groupMimeType.toUpperCase()))
            .orElse(false);
    }
    
    /**
     * Get the file type of a resource file
     * @param file the resource
     * @return the {@link FileType}
     */
    public FileType getFileType(Resource file)
    {
        Optional<String> group = SolrResourceGroupedMimeTypes.getGroup(file.getMimeType());
        
        return group
                .map(groupMimeType -> FileType.valueOf(groupMimeType.toUpperCase()))
                .orElse(FileType.UNKNOWN);
    }
    
    /**
     * Get the file type of a resource from its name
     * @param filename the resource's name
     * @return the {@link FileType}
     */
    public FileType getFileType(String filename)
    {
        String mimeType = _cocoonContext.getMimeType(filename);
        Optional<String> group = SolrResourceGroupedMimeTypes.getGroup(mimeType);
        
        return group
                .map(groupMimeType -> FileType.valueOf(groupMimeType.toUpperCase()))
                .orElse(FileType.UNKNOWN);
    }

    /**
     * Get user avatar
     * @param userIdentity the user identity
     * @param lang the lang for user's content
     * @param size the size in pixel of image
     * @return the avatar url
     */
    public String getAvatar(UserIdentity userIdentity, String lang, int size)
    {
        StringBuilder sb = new StringBuilder();
        
        String siteName = WebHelper.getSiteName(_getRequest());
        sb.append(_prefixHandler.getUriPrefix(siteName))
            .append("/_plugins/user-directory/user/")
            .append(userIdentity.getPopulationId())
            .append("/")
            .append(URIUtils.encodePath(userIdentity.getLogin()))
            .append("/image_")
            .append(size)
            .append("?lang=")
            .append(lang);
        
        return sb.toString();
    }
    
    private Request _getRequest()
    {
        return ContextHelper.getRequest(_context);
    }
    
    /**
     * Get current project from request
     * @return the current project or null if not found.
     */
    public Project getProjectFromRequest()
    {
        Request request = _getRequest();
        String projectName = (String) request.getAttribute(WorkspacesConstants.REQUEST_ATTR_PROJECT_NAME);
        
        if (StringUtils.isNotEmpty(projectName))
        {
            return _projectManager.getProject(projectName);
        }
        
        String siteName = WebHelper.getSiteName(request);
        if (StringUtils.isNotEmpty(siteName) && _siteManager.hasSite(siteName))
        {
            Site site = _siteManager.getSite(siteName);
            List<Project> projects = _projectManager.getProjectsForSite(site);
            if (!projects.isEmpty())
            {
                return projects.get(0);
            }
        }
        
        return null;
    }
    
    /**
     * Handle tags for the edition of a TaggableAmetysObject
     * @param taggableAmetysObject the object to edit
     * @param tags the tags
     * @param moduleRoot the module root, used to check tag creation rights if needed
     * @return the created tags
     */
    public List<Map<String, Object>> handleTags(TaggableAmetysObject taggableAmetysObject, List<Object> tags, TraversableAmetysObject moduleRoot)
    {
        List<String> createdTags = new ArrayList<>();
        List<Map<String, Object>> createdTagsJson = new ArrayList<>();
        // Tag new tags
        for (Object tag : tags)
        {
            // Tag doesn't exist so create the tag
            if (tag instanceof Map)
            {
                if (_rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_ADD_TAG, moduleRoot) != RightResult.RIGHT_ALLOW)
                {
                    throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to handle tags without convenient right.");
                }
                @SuppressWarnings("unchecked")
                String tagText = (String) ((Map<String, Object>) tag).get("text");
                List<Map<String, Object>> newTags = _projectTagsDAO.addTags(new String[] {tagText});
                String newTag = (String) newTags.get(0).get("name");
                taggableAmetysObject.tag(newTag);
                createdTags.add(newTag);
                createdTagsJson.addAll(newTags);
            }
            else
            {
                taggableAmetysObject.tag((String) tag);
            }
        }
        // Untag unused tags
        for (String tag : taggableAmetysObject.getTags())
        {
            if (!tags.contains(tag) && !createdTags.contains(tag))
            {
                taggableAmetysObject.untag(tag);
            }
        }
        return createdTagsJson;
    }

    /**
     * Get the lang for this project, or en if none found
     * @param project the project to check
     * @return a lang in this project
     */
    public String getLang(Project project)
    {
        /*
         * get the site, then the sitemaps (transformed back to a stream of sitemap)
         * then get the name
         * return the 1st one, or "en" by default if none is available, or if the site could not be found
         */
        
        Site site = project.getSite();
        
        if (site != null)
        {
            return site.getSitemaps()
                       .stream()
                       .filter(Objects::nonNull)
                       .map(Sitemap::getName)
                       .findFirst()
                       .orElse("en");
        }
        else
        {
            return "en";
        }
    }
}
