/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.minisite;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.ametys.core.observation.Event;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;

/**
 * {@link ActivityType} implementation for mini-site related events
 */
public class MinisiteActivityType extends AbstractWorkspacesActivityType
{
    /** data name for the page title */
    public static final String PAGE_TITLE = "pageTitle";
    /** data name for the page id */
    public static final String PAGE_ID = "pageId";
    /** Constant for activity's category */
    public static final String ACTIVITY_CATEGORY_MINISITE = "minisite";
    
    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);
        
        activity.setValue(PROJECT_CATEGORY, ACTIVITY_CATEGORY_MINISITE);
        
        Page page = _getPage(parameters);
        if (page != null)
        {
            activity.setValue(PAGE_ID, page.getId());
            activity.setValue(PAGE_TITLE, page.getTitle());
        }
        else
        {
            String pageId = (String) parameters.get(ObservationConstants.ARGS_PAGE_ID);
            if (pageId != null)
            {
                activity.setValue(PAGE_ID, pageId);
            }
            String pageTitle = (String) parameters.get(org.ametys.plugins.workspaces.ObservationConstants.ARGS_PAGE_TITLE);
            if (pageTitle != null)
            {
                activity.setValue(PAGE_TITLE, pageTitle);
            }
        }
    }
    
    private Page _getPage(Map<String, Object> parameters)
    {
        SitemapElement sitemapElement = (SitemapElement) parameters.get(ObservationConstants.ARGS_SITEMAP_ELEMENT);
        if (sitemapElement != null && sitemapElement instanceof Page page)
        {
            return page;
        }
        else
        {
            return (Page) parameters.get(ObservationConstants.ARGS_PAGE);
        }
    }
    
    @Override
    public Map<String, Object> mergeActivities(List<Activity> activities)
    {
        Map<String, Object> mergedActivities = super.mergeActivities(activities);
        
        List<Map<String, Object>> mergedPages = new ArrayList<>();
        
        List<String> knownPages = new ArrayList<>();
        
        for (Activity event : activities)
        {
            String pageId = event.getValue(PAGE_ID);
            if (!knownPages.contains(pageId))
            {
                knownPages.add(pageId);
                Map<String, Object> pageInfo = new HashMap<>();
                
                pageInfo.put(PAGE_ID, event.getValue(PAGE_ID));
                pageInfo.put(PAGE_TITLE, event.getValue(PAGE_TITLE));
                
                mergedPages.add(pageInfo);
            }
        }
        mergedActivities.put("pages", mergedPages);
        mergedActivities.put("amount", mergedPages.size());
        
        return mergedActivities;
    }
    
    @Override
    public Expression getFilterPatternExpression(String pattern)
    {
        return new StringExpression(PAGE_TITLE, Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true));
    }
    
    @Override
    public Project getProjectFromEvent(Event event)
    {
        Map<String, Object> args = event.getArguments();
        
        return (Project) args.get(org.ametys.plugins.workspaces.ObservationConstants.ARGS_PROJECT);
    }
}
