/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.jcr;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.messagingconnector.EventRecurrenceTypeEnum;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.tag.TaggableAmetysObjectHelper;
import org.ametys.plugins.workflow.repository.WorkflowAwareAmetysObject;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventAttendee;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventOccurrence;
import org.ametys.plugins.workspaces.calendars.events.ModifiableCalendarEvent;
import org.ametys.plugins.workspaces.calendars.helper.RecurrentEventHelper;

/**
 * Default implementation of an {@link CalendarEvent}, backed by a JCR node.<br>
 */
public class JCRCalendarEvent extends DefaultTraversableAmetysObject<JCRCalendarEventFactory> implements ModifiableCalendarEvent, WorkflowAwareAmetysObject
{

    /** Attribute name for event author*/
    public static final String ATTRIBUTE_CREATOR = "creator";
    
    /** Attribute name for event lastModified*/
    public static final String ATTRIBUTE_CREATION = "creationDate";

    /** Attribute name for event last contributor*/
    public static final String ATTRIBUTE_CONTRIBUTOR = "contributor";
    
    /** Attribute name for event lastModified*/
    public static final String ATTRIBUTE_MODIFIED = "lastModified";

    /** Attribute name for event title */
    public static final String ATTRIBUTE_TITLE = "title";
    
    /** Attribute name for event description*/
    public static final String ATTRIBUTE_DESC = "description";
    
    /** Attribute name for event keywords' */
    public static final String ATTRIBUTE_KEYWORDS = "keywords";
    
    /** Attribute name for event location*/
    public static final String ATTRIBUTE_LOCATION = "location";
    
    /** Attribute name for event startDate*/
    public static final String ATTRIBUTE_START_DATE = "startDate";
    
    /** Attribute name for event endDate*/
    public static final String ATTRIBUTE_END_DATE = "endDate";
    
    /** Attribute name for event date sone*/
    public static final String ATTRIBUTE_DATE_ZONE = "dateZone";
        
    /** Attribute name for event fullDay*/
    public static final String ATTRIBUTE_FULL_DAY = "fullDay";
    
    /** Attribute name for event recurrence type */
    public static final String ATTRIBUTE_RECURRENCE_TYPE = "recurrenceType";
    
    /** Attribute name for event until date */
    public static final String ATTRIBUTE_UNTIL_DATE = "untilDate";
    
    /** Attribute name for event excluded date */
    public static final String ATTRIBUTE_EXCLUDED_DATE = "excludedDate";
    
    /** Attribute name for event organiser */
    public static final String ATTRIBUTE_ORGANISER = "organiser";
    
    /** Property name for attendee population * */
    public static final String PROPERTY_ATTENDEE_POPULATION = "populationId";
    
    /** Property name for attendee login * */
    public static final String PROPERTY_ATTENDEE_LOGIN = "login";
    
    /** Property name for attendee email * */
    public static final String PROPERTY_ATTENDEE_EMAIL = "email";
    
    /** Property name for attendee external * */
    public static final String PROPERTY_ATTENDEE_EXTERNAL = "external";
    
    /** Property name for attendee mandatory * */
    public static final String PROPERTY_ATTENDEE_MANDATORY = "mandatory";
    
    /** Name of the node for attendees * */
    public static final String NODE_ATTENDEES_NAME = RepositoryConstants.NAMESPACE_PREFIX + ":calendar-event-attendees";
    
    /** Name of the node for attendee * */
    public static final String NODE_ATTENDEE_NAME = RepositoryConstants.NAMESPACE_PREFIX + ":calendar-event-attendee";
    
    
    /** Property's name for workflow id */
    public static final String PROPERTY_WORKFLOW_ID = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":workflowId";
    
    /** Property's name for resources */
    public static final String ATTRIBUTE_RESOURCES = "resources";
    
    /**
     * Creates an {@link JCRCalendarEvent}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public JCRCalendarEvent(Node node, String parentPath, JCRCalendarEventFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    public Calendar getCalendar()
    {
        return getParent();
    }

    @Override
    public String getTitle()
    {
        return getValue(ATTRIBUTE_TITLE);
    }
    
    @Override
    public String getDescription()
    {
        return getValue(ATTRIBUTE_DESC);
    }
    
    @Override
    public String getLocation()
    {
        return getValue(ATTRIBUTE_LOCATION, true, null);
    }

    public void tag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.tag(this, tag);
    }

    public void untag(String tag) throws AmetysRepositoryException
    {
        TaggableAmetysObjectHelper.untag(this, tag);
    }

    public Set<String> getTags() throws AmetysRepositoryException
    {
        return TaggableAmetysObjectHelper.getTags(this);
    }
    
    @Override
    public ZonedDateTime getStartDate()
    {
        return getValue(ATTRIBUTE_START_DATE);
    }

    @Override
    public ZonedDateTime getEndDate()
    {
        return getValue(ATTRIBUTE_END_DATE);
    }

    @Override
    public ZoneId getZone()
    {
        if (hasValue(ATTRIBUTE_DATE_ZONE))
        {
            String zoneId = getValue(ATTRIBUTE_DATE_ZONE);
            if (ZoneId.of(zoneId) != null)
            {
                return ZoneId.of(zoneId);
            }
        }
        
        // For legacy purposes: old events won't have any zone, in this case, use system default zone id
        return ZoneId.systemDefault();
    }
    
    @Override
    public Boolean getFullDay()
    {
        return getValue(ATTRIBUTE_FULL_DAY);
    }
    
    @Override
    public UserIdentity getCreator()
    {
        return getValue(ATTRIBUTE_CREATOR);
    }

    @Override
    public ZonedDateTime getCreationDate()
    {
        return getValue(ATTRIBUTE_CREATION);
    }

    @Override
    public UserIdentity getLastContributor()
    {
        return getValue(ATTRIBUTE_CONTRIBUTOR);
    }

    @Override
    public ZonedDateTime getLastModified()
    {
        return getValue(ATTRIBUTE_MODIFIED);
    }
    
    @Override
    public EventRecurrenceTypeEnum getRecurrenceType()
    {
        String recurrenceType = getValue(ATTRIBUTE_RECURRENCE_TYPE, true, EventRecurrenceTypeEnum.NEVER.toString());
        EventRecurrenceTypeEnum recurrenceEnum = EventRecurrenceTypeEnum.valueOf(recurrenceType);
        return recurrenceEnum;
    }
    
    @Override
    public Boolean isRecurrent()
    {
        return !getRecurrenceType().equals(EventRecurrenceTypeEnum.NEVER);
    }
    
    @Override
    public ZonedDateTime getRepeatUntil()
    {
        return getValue(ATTRIBUTE_UNTIL_DATE);
    }
    
    private ZonedDateTime _getUntilDate()
    {
        ZonedDateTime untilDate = this.getRepeatUntil();
        // until date must be included in list of occurrences.
        // For that purpose, until date is used here as a threshold value that
        // must be set to the start of the next day
        if (untilDate != null)
        {
            untilDate = untilDate.plusDays(1);
        }
        return untilDate;
    }
    
    @Override
    public List<ZonedDateTime> getExcludedOccurences()
    {
        ZonedDateTime[] excludedOccurences = getValue(ATTRIBUTE_EXCLUDED_DATE, false, new ZonedDateTime[0]);
        return Arrays.asList(excludedOccurences);
    }
    
    @Override
    public UserIdentity getOrganiser()
    {
        return getValue(ATTRIBUTE_ORGANISER);
    }

    @Override
    public List<String> getResources()
    {
        String[] resources = getValue(ATTRIBUTE_RESOURCES, false, new String[0]);
        return Arrays.asList(resources);
    }
    
    @Override
    public void setTitle(String title)
    {
        setValue(ATTRIBUTE_TITLE, title);
    }
    
    @Override
    public void setDescription(String desc)
    {
        setValue(ATTRIBUTE_DESC, desc);
    }
    
    @Override
    public void setLocation(String location)
    {
        setValue(ATTRIBUTE_LOCATION, location);
    }

    
    @Override
    public void setStartDate(ZonedDateTime startDate)
    {
        setValue(ATTRIBUTE_START_DATE, startDate);
    }

    @Override
    public void setEndDate(ZonedDateTime endDate)
    {
        setValue(ATTRIBUTE_END_DATE, endDate);     
    }

    public void setZone(ZoneId dateZone)
    {
        setValue(ATTRIBUTE_DATE_ZONE, dateZone.getId());    
    }

    @Override
    public void setFullDay(Boolean fullDay)
    {
        setValue(ATTRIBUTE_FULL_DAY, fullDay);
    }
    
    @Override
    public void setCreator(UserIdentity user)
    {
        try
        {
            Node creatorNode = null;
            if (getNode().hasNode(RepositoryConstants.NAMESPACE_PREFIX + ':' + ATTRIBUTE_CREATOR))
            {
                creatorNode = getNode().getNode(RepositoryConstants.NAMESPACE_PREFIX + ':' + ATTRIBUTE_CREATOR);
            }
            else
            {
                creatorNode = getNode().addNode(RepositoryConstants.NAMESPACE_PREFIX + ':' + ATTRIBUTE_CREATOR, RepositoryConstants.USER_NODETYPE);
            }
            creatorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login", user.getLogin());
            creatorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population", user.getPopulationId());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the creator property.", e);
        }
    }

    @Override
    public void setCreationDate(ZonedDateTime date)
    {
        setValue(ATTRIBUTE_CREATION, date);
    }

    @Override
    public void setLastContributor(UserIdentity user)
    {
        try
        {
            Node creatorNode = null;
            if (getNode().hasNode(RepositoryConstants.NAMESPACE_PREFIX + ':' + ATTRIBUTE_CONTRIBUTOR))
            {
                creatorNode = getNode().getNode(RepositoryConstants.NAMESPACE_PREFIX + ':' + ATTRIBUTE_CONTRIBUTOR);
            }
            else
            {
                creatorNode = getNode().addNode(RepositoryConstants.NAMESPACE_PREFIX + ':' + ATTRIBUTE_CONTRIBUTOR, RepositoryConstants.USER_NODETYPE);
            }
            creatorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login", user.getLogin());
            creatorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population", user.getPopulationId());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the contributor property.", e);
        }
    }

    @Override
    public void setLastModified(ZonedDateTime date)
    {
        setValue(ATTRIBUTE_MODIFIED, date);
    }
        
    @Override
    public void setRecurrenceType(String recurrenceType)
    {
        setValue(ATTRIBUTE_RECURRENCE_TYPE, recurrenceType);
    }
    
    @Override
    public void setRepeatUntil(ZonedDateTime untilDate)
    {
        if (untilDate == null)
        {
            if (hasValue(ATTRIBUTE_UNTIL_DATE))
            {
                removeValue(ATTRIBUTE_UNTIL_DATE);
            }
        }
        else
        {
            setValue(ATTRIBUTE_UNTIL_DATE, untilDate);
        }
    }
    
    @Override
    public void setExcludedOccurrences(List<ZonedDateTime> excludedOccurrences)
    {
        setValue(ATTRIBUTE_EXCLUDED_DATE, excludedOccurrences.toArray(new ZonedDateTime[excludedOccurrences.size()]));
    }
    
    @Override
    public List<CalendarEventOccurrence> getOccurrences(ZonedDateTime startDate, ZonedDateTime endDate)
    {
        Optional<CalendarEventOccurrence> optionalEvent = getFirstOccurrence(startDate);
        if (optionalEvent.isPresent() && optionalEvent.get().getStartDate().isBefore(endDate))
        {
            return RecurrentEventHelper.getOccurrences(optionalEvent.get().getStartDate(), endDate, optionalEvent.get().getStartDate(), 
                    optionalEvent.get().getStartDate(), getRecurrenceType(), getExcludedOccurences(), getZone(), _getUntilDate())
                    .stream()
                    .map(occurrenceStartDate -> new CalendarEventOccurrence(this, occurrenceStartDate))
                    .collect(Collectors.toList());
        }
        else
        {
            return new ArrayList<>();
        }
    }
    
    @Override
    public Optional<CalendarEventOccurrence> getFirstOccurrence(ZonedDateTime date)
    {
        ZonedDateTime eventStartDate = getStartDate();
        ZonedDateTime eventEndDate = getEndDate();
        
        if (getFullDay())
        {
            eventEndDate = eventEndDate.plusDays(1);
        }
        
        if (eventEndDate.isAfter(date) || eventEndDate.isEqual(date))
        {
            // Return the event himself
            return Optional.of(new CalendarEventOccurrence(this, eventStartDate));
        }
        else if (this.isRecurrent())
        {
            ZonedDateTime untilDate = _getUntilDate();
            
            long eventDuringTime = ChronoUnit.SECONDS.between(eventStartDate, eventEndDate);
            
            ZonedDateTime endDate = eventEndDate;
            ZonedDateTime startDate = eventStartDate;
            while (startDate != null && endDate.isBefore(date))
            {
                startDate = RecurrentEventHelper.getNextDate(getRecurrenceType(), getStartDate().withZoneSameInstant(getZone()), startDate.withZoneSameInstant(getZone()));
                if (startDate != null)
                {
                    endDate = startDate.plusSeconds(eventDuringTime);
                }
            }
            if (untilDate == null || untilDate.isAfter(startDate))
            {
                return Optional.of(new CalendarEventOccurrence(this, startDate));
            }
        }
        return Optional.empty();
    }
    
    @Override
    public Optional<CalendarEventOccurrence> getNextOccurrence(CalendarEventOccurrence occurrence)
    {
        ZonedDateTime untilDate = _getUntilDate();
        ZonedDateTime nextDate = RecurrentEventHelper.getNextDate(getRecurrenceType(), getStartDate().withZoneSameInstant(getZone()), occurrence.getStartDate().withZoneSameInstant(getZone()));
        if (nextDate != null && (untilDate == null || untilDate.isAfter(nextDate)))
        {
            return Optional.of(new CalendarEventOccurrence(this, nextDate));
        }
        
        return Optional.empty();
    }
    
    @Override
    public long getWorkflowId() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_WORKFLOW_ID).getLong();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get workflowId property", e);
        }
    }
    
    @Override
    public void setWorkflowId(long workflowId) throws AmetysRepositoryException
    {
        Node node = getNode();
        
        try
        {
            node.setProperty(PROPERTY_WORKFLOW_ID, workflowId);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set workflowId property", e);
        }
    }
    
    @Override
    public long getCurrentStepId()
    {
        throw new UnsupportedOperationException();
    }
    
    @Override
    public void setCurrentStepId(long stepId)
    {
        throw new UnsupportedOperationException();
    }
    
    @Override
    public void setOrganiser(UserIdentity user)
    {
        try
        {
            Node creatorNode = null;
            if (getNode().hasNode(RepositoryConstants.NAMESPACE_PREFIX + ':' + ATTRIBUTE_ORGANISER))
            {
                creatorNode = getNode().getNode(RepositoryConstants.NAMESPACE_PREFIX + ':' + ATTRIBUTE_ORGANISER);
            }
            else
            {
                creatorNode = getNode().addNode(RepositoryConstants.NAMESPACE_PREFIX + ':' + ATTRIBUTE_ORGANISER, RepositoryConstants.USER_NODETYPE);
            }
            creatorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login", user.getLogin());
            creatorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population", user.getPopulationId());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the organiser property.", e);
        }
    }
    
    /**
     * Get attendees to the event
     * @throws RepositoryException if an error occurred
     * @return the attendees
     */
    public List<CalendarEventAttendee> getAttendees() throws RepositoryException 
    {
        List<CalendarEventAttendee> attendees = new ArrayList<>();
        
        Node calendarEventNode = getNode();
        if (calendarEventNode.hasNode(NODE_ATTENDEES_NAME))
        {
            Node attendeesNode = calendarEventNode.getNode(NODE_ATTENDEES_NAME);
            NodeIterator nodes = attendeesNode.getNodes();
            while (nodes.hasNext())
            {
                Node attendeeNode = (Node) nodes.next();
                CalendarEventAttendee attendee = new CalendarEventAttendee();
                
                boolean isExternal = attendeeNode.getProperty(PROPERTY_ATTENDEE_EXTERNAL).getBoolean();
                if (isExternal)
                {
                    attendee.setEmail(attendeeNode.getProperty(PROPERTY_ATTENDEE_EMAIL).getString());
                }
                else
                {
                    attendee.setLogin(attendeeNode.getProperty(PROPERTY_ATTENDEE_LOGIN).getString());
                    attendee.setPopulationId(attendeeNode.getProperty(PROPERTY_ATTENDEE_POPULATION).getString());
                }
                
                attendee.setIsExternal(isExternal);
                attendee.setIsMandatory(attendeeNode.getProperty(PROPERTY_ATTENDEE_MANDATORY).getBoolean());
                
                attendees.add(attendee);
            }
        }
        
        return attendees;
    }
    
    /**
     * Set attendees to the event
     * @param attendees the list of attendees
     * @throws RepositoryException if an error occurred
     */
    public void setAttendees(List<CalendarEventAttendee> attendees) throws RepositoryException 
    {
        Node calendarEventNode = getNode();
        
        if (calendarEventNode.hasNode(NODE_ATTENDEES_NAME))
        {
            calendarEventNode.getNode(NODE_ATTENDEES_NAME).remove();
        }
        
        Node attendeesNode = calendarEventNode.addNode(NODE_ATTENDEES_NAME, "ametys:unstructured");
        for (CalendarEventAttendee attendee : attendees)
        {
            // Create new attendee
            Node attendeeNode = attendeesNode.addNode(NODE_ATTENDEE_NAME, "ametys:unstructured");
            if (attendee.isExternal())
            {
                attendeeNode.setProperty(PROPERTY_ATTENDEE_EMAIL, attendee.getEmail());
            }
            else
            {
                attendeeNode.setProperty(PROPERTY_ATTENDEE_POPULATION, attendee.getPopulationId());
                attendeeNode.setProperty(PROPERTY_ATTENDEE_LOGIN, attendee.getLogin());
            }
            
            attendeeNode.setProperty(PROPERTY_ATTENDEE_EXTERNAL, attendee.isExternal());
            attendeeNode.setProperty(PROPERTY_ATTENDEE_MANDATORY, attendee.isMandatory());
        }
    }

    @Override
    public void setResources(List<String> resources)
    {
        setValue(ATTRIBUTE_RESOURCES, resources.toArray(new String[resources.size()]));
    }
    
    public ModifiableIndexableDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, _getFactory().getCalendarEventModel());
    }
}
