/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.task;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.messagingconnector.EventRecurrenceTypeEnum;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventOccurrence;
import org.ametys.plugins.workspaces.tasks.Task;

/**
 * Object for a task calendar event
 */
public class TaskCalendarEvent implements CalendarEvent
{
    private Calendar _calendar;
    private Task _task;
    
    /**
     * The constructor
     * @param calendar the calendar
     * @param task the task
     */
    public TaskCalendarEvent(Calendar calendar, Task task)
    {
        _calendar = calendar;
        _task = task;
    }
    
    /**
     * Return the task link to the calendar event
     * @return the task
     */
    public Task getTask()
    {
        return _task;
    }
    
    public void tag(String tag) throws AmetysRepositoryException
    {
        throw new UnsupportedOperationException("Can't call method #tag for a TaskCalendarEvent");
    }

    public void untag(String tag) throws AmetysRepositoryException
    {
        throw new UnsupportedOperationException("Can't call method #untag for a TaskCalendarEvent");
    }

    public Set<String> getTags() throws AmetysRepositoryException
    {
        return _task.getTags();
    }

    public String getName() throws AmetysRepositoryException
    {
        return _task.getName();
    }

    public String getPath() throws AmetysRepositoryException
    {
        return _task.getPath();
    }

    public String getId() throws AmetysRepositoryException
    {
        return _task.getId();
    }

    public <A extends AmetysObject> A getParent() throws AmetysRepositoryException
    {
        return _task.getParent();
    }

    public String getParentPath() throws AmetysRepositoryException
    {
        return _task.getParentPath();
    }

    public Calendar getCalendar()
    {
        return _calendar;
    }
    
    public String getTitle()
    {
        return _task.getLabel();
    }

    public String getDescription()
    {
        return _task.getDescription();
    }

    public String getLocation()
    {
        return null;
    }

    public ZonedDateTime getStartDate()
    {
        LocalDate date = _task.getStartDate() != null 
                ? _task.getStartDate()
                : _task.getDueDate();
        return DateUtils.asZonedDateTime(date, getZone());
    }

    public ZonedDateTime getEndDate()
    {
        LocalDate date = _task.getDueDate() != null 
                ? _task.getDueDate()
                : _task.getStartDate();
        return DateUtils.asZonedDateTime(date, getZone());
    }

    public ZoneId getZone()
    {
        return ZoneId.systemDefault();
    }

    public Boolean getFullDay()
    {
        return true;
    }

    public UserIdentity getCreator()
    {
        return _task.getAuthor();
    }

    public ZonedDateTime getCreationDate()
    {
        return _task.getCreationDate();
    }

    public UserIdentity getLastContributor()
    {
        return _task.getAuthor();
    }

    public ZonedDateTime getLastModified()
    {
        return _task.getLastModified();
    }

    public EventRecurrenceTypeEnum getRecurrenceType()
    {
        return EventRecurrenceTypeEnum.NEVER;
    }

    public Boolean isRecurrent()
    {
        return false;
    }

    public ZonedDateTime getRepeatUntil()
    {
        return null;
    }

    public List<ZonedDateTime> getExcludedOccurences()
    {
        return List.of();
    }

    public List<CalendarEventOccurrence> getOccurrences(ZonedDateTime startDate, ZonedDateTime endDate)
    {
        Optional<CalendarEventOccurrence> optionalEvent = getFirstOccurrence(startDate);
        return (optionalEvent.isPresent() && optionalEvent.get().getStartDate().isBefore(endDate))
            ? List.of(optionalEvent.get())
            : List.of();
    }

    public Optional<CalendarEventOccurrence> getFirstOccurrence(ZonedDateTime date)
    {
        ZonedDateTime eventStartDate = getStartDate();
        ZonedDateTime eventEndDate = getEndDate();
        
        if (getFullDay())
        {
            eventEndDate = eventEndDate.plusDays(1);
        }
        
        return (eventEndDate.isAfter(date) || eventEndDate.isEqual(date))
            ? Optional.of(new CalendarEventOccurrence(this, eventStartDate))
            : Optional.empty();
    }

    public Optional<CalendarEventOccurrence> getNextOccurrence(CalendarEventOccurrence occurrence)
    {
        return Optional.empty();
    }

    public UserIdentity getOrganiser()
    {
        return _task.getAuthor();
    }

    public List<String> getResources()
    {
        return List.of(_calendar.getId());
    }
}
