/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.cmis;

import java.io.InputStream;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.chemistry.opencmis.commons.PropertyIds;
import org.apache.chemistry.opencmis.commons.data.AllowableActions;
import org.apache.chemistry.opencmis.commons.data.ContentStream;
import org.apache.chemistry.opencmis.commons.data.FailedToDeleteData;
import org.apache.chemistry.opencmis.commons.data.ObjectData;
import org.apache.chemistry.opencmis.commons.data.ObjectInFolderList;
import org.apache.chemistry.opencmis.commons.data.ObjectParentData;
import org.apache.chemistry.opencmis.commons.data.Properties;
import org.apache.chemistry.opencmis.commons.data.PropertyData;
import org.apache.chemistry.opencmis.commons.data.RepositoryInfo;
import org.apache.chemistry.opencmis.commons.definitions.PropertyDefinition;
import org.apache.chemistry.opencmis.commons.definitions.TypeDefinition;
import org.apache.chemistry.opencmis.commons.definitions.TypeDefinitionList;
import org.apache.chemistry.opencmis.commons.enums.Action;
import org.apache.chemistry.opencmis.commons.enums.BaseTypeId;
import org.apache.chemistry.opencmis.commons.enums.CapabilityAcl;
import org.apache.chemistry.opencmis.commons.enums.CapabilityChanges;
import org.apache.chemistry.opencmis.commons.enums.CapabilityContentStreamUpdates;
import org.apache.chemistry.opencmis.commons.enums.CapabilityJoin;
import org.apache.chemistry.opencmis.commons.enums.CapabilityRenditions;
import org.apache.chemistry.opencmis.commons.enums.CmisVersion;
import org.apache.chemistry.opencmis.commons.enums.Updatability;
import org.apache.chemistry.opencmis.commons.enums.VersioningState;
import org.apache.chemistry.opencmis.commons.exceptions.CmisBaseException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisConstraintException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisContentAlreadyExistsException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisInvalidArgumentException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisObjectNotFoundException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisRuntimeException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisStorageException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisStreamNotSupportedException;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.AllowableActionsImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.ContentStreamImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.FailedToDeleteDataImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.ObjectDataImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.ObjectInFolderDataImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.ObjectInFolderListImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.ObjectParentDataImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.PartialContentStreamImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.PropertiesImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.PropertyBooleanImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.PropertyDateTimeImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.PropertyIdImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.PropertyIntegerImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.PropertyStringImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.RepositoryCapabilitiesImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.RepositoryInfoImpl;
import org.apache.chemistry.opencmis.commons.impl.server.ObjectInfoImpl;
import org.apache.chemistry.opencmis.commons.server.CallContext;
import org.apache.chemistry.opencmis.commons.server.ObjectInfoHandler;
import org.apache.chemistry.opencmis.commons.spi.Holder;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.resources.ModifiableResource;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper.ResourceOperationMode;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper.ResourceOperationResult;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Helper class to retreive CMIS objects
 *
 */
public class CmisRepository
{
    private static final String USER_UNKNOWN = "<unknown>";
    
    /**
     * retreive informations for a repository
     * @param context call context
     * @param project project
     * @param factory factory
     * @return RepositoryInfo
     */
    public RepositoryInfo getRepositoryInfo(CallContext context, Project project, CmisServiceFactory factory)
    {
        String id = project.getId();
        String name = project.getName();
        if (id == null || id.trim().length() == 0 || name == null || name.trim().length() == 0)
        {
            throw new CmisInvalidArgumentException("Invalid repository!");
        }

        // set up repository infos
        return createRepositoryInfo(project, factory, CmisVersion.CMIS_1_1);
    }
    
    private RepositoryInfo createRepositoryInfo(Project project, CmisServiceFactory factory, CmisVersion cmisVersion)
    {
        assert cmisVersion != null;

        RepositoryInfoImpl repositoryInfo = new RepositoryInfoImpl();

        String id = project.getId();
        ModifiableResourceCollection documentRoot = getRoot(project, factory);
        
        String root = "";
        if (documentRoot != null)
        {
            root = documentRoot.getId();
        }
        
        id = URIUtils.encodeParameter(URIUtils.encodeParameter(id));
        root = URIUtils.encodeParameter(URIUtils.encodeParameter(root));
        
        repositoryInfo.setId(id);
        repositoryInfo.setName(project.getName());
        repositoryInfo.setDescription(project.getDescription());

        repositoryInfo.setCmisVersionSupported(cmisVersion.value());

        repositoryInfo.setProductName("Ametys CMIS Server");
        repositoryInfo.setProductVersion("1.0");
        repositoryInfo.setVendorName("Ametys");

        repositoryInfo.setRootFolder(root);

        repositoryInfo.setThinClientUri("");
        repositoryInfo.setChangesIncomplete(false);
        RepositoryCapabilitiesImpl capabilities = new RepositoryCapabilitiesImpl();
        capabilities.setCapabilityAcl(CapabilityAcl.NONE);
        capabilities.setAllVersionsSearchable(false);
        capabilities.setCapabilityJoin(CapabilityJoin.NONE);
        capabilities.setSupportsMultifiling(false);
        capabilities.setSupportsUnfiling(false);
        capabilities.setSupportsVersionSpecificFiling(false);
        capabilities.setIsPwcSearchable(false);
        capabilities.setIsPwcUpdatable(false);
        //capabilities.setCapabilityQuery(CapabilityQuery.METADATAONLY);
        capabilities.setCapabilityChanges(CapabilityChanges.NONE);
        capabilities.setCapabilityContentStreamUpdates(CapabilityContentStreamUpdates.ANYTIME);
        capabilities.setSupportsGetDescendants(false);
        capabilities.setSupportsGetFolderTree(false);
        capabilities.setCapabilityRendition(CapabilityRenditions.NONE);

        repositoryInfo.setCapabilities(capabilities);
        
        return repositoryInfo;
    }
    
    /**
     * getTypeDefinition
     * @param context call context
     * @param typeId type ID
     * @param factory factory
     * @return the type definition
     */
    public TypeDefinition getTypeDefinition(CallContext context, String typeId, CmisServiceFactory factory) 
    {
        //checkUser(context, false);

        return factory.getTypeManager().getTypeDefinition(context, typeId);
    }
    
    /**
     * getTypeChildren
     * @param context context
     * @param typeId typeId
     * @param includePropertyDefinitions includePropertyDefinitions
     * @param maxItems maxItems
     * @param skipCount skipCount
     * @param factory factory
     * @return the type definition list
     */
    public TypeDefinitionList getTypeChildren(CallContext context, String typeId, Boolean includePropertyDefinitions,
            BigInteger maxItems, BigInteger skipCount, CmisServiceFactory factory) 
    {
        //checkUser(context, false);

        return factory.getTypeManager().getTypeChildren(context, typeId, includePropertyDefinitions, maxItems, skipCount);
    }
    
    /**
     * getObject
     * @param context context
     * @param project Project
     * @param objectId objectId
     * @param versionServicesId versionServicesId
     * @param filter filter
     * @param includeAllowableActions includeAllowableActions
     * @param includeAcl includeAcl
     * @param objectInfos objectInfos
     * @param factory factory
     * @return ObjectData ObjectData
     */
    public ObjectData getObject(CallContext context, Project project, String objectId, String versionServicesId, String filter,
            Boolean includeAllowableActions, Boolean includeAcl, ObjectInfoHandler objectInfos, 
            CmisServiceFactory factory)
    {
        // check id
        if (objectId == null && versionServicesId == null)
        {
            throw new CmisInvalidArgumentException("Object Id must be set.");
        }
        
        AmetysObject ametysObject = factory.getResolver().resolveById(objectId);
        
        boolean userReadOnly = false;

        // set defaults if values not set
        boolean iaa = CmisUtils.getBooleanParameter(includeAllowableActions, false);
        boolean iacl = false; //CmisUtils.getBooleanParameter(includeAcl, false);

        // split filter
        Set<String> filterCollection = CmisUtils.splitFilter(filter);

        // gather properties
        
        return compileObjectData(context, ametysObject, project, filterCollection, iaa, iacl, userReadOnly, objectInfos, factory);
    }
    
    
    private ObjectData compileObjectData(CallContext context, AmetysObject ametysObject, Project project, Set<String> filter,
            boolean includeAllowableActions, boolean includeAcl, boolean userReadOnly, ObjectInfoHandler objectInfos,
            CmisServiceFactory factory)
    {
        ObjectDataImpl result = new ObjectDataImpl();
        ObjectInfoImpl objectInfo = new ObjectInfoImpl();

        result.setProperties(compileProperties(context, ametysObject, project, filter, objectInfo, factory));

        if (includeAllowableActions)
        {
            result.setAllowableActions(compileAllowableActions(ametysObject, project, factory, userReadOnly));
        }

        if (includeAcl)
        {
            result.setIsExactAcl(true);
        }
        result.setIsExactAcl(true);

        if (context.isObjectInfoRequired())
        {
            objectInfo.setObject(result);
            objectInfos.addObjectInfo(objectInfo);
        }

        return result;
    }
    
    private Properties compileProperties(CallContext context, AmetysObject ametysObject, Project project, Set<String> orgfilter,
            ObjectInfoImpl objectInfo, CmisServiceFactory factory)
    {
        // copy filter
        Set<String> filter = orgfilter == null ? null : new HashSet<>(orgfilter);

        // find base type
        String typeId = null;
        Boolean isFolder = true;
        
        if (ametysObject != null && ametysObject instanceof ModifiableResource)
        {
            isFolder = false;
            typeId = BaseTypeId.CMIS_DOCUMENT.value();
            objectInfo.setBaseType(BaseTypeId.CMIS_DOCUMENT);
            objectInfo.setTypeId(typeId);
            objectInfo.setHasAcl(false);
            objectInfo.setHasContent(true);
            objectInfo.setHasParent(true);
            objectInfo.setVersionSeriesId(null);
            objectInfo.setIsCurrentVersion(true);
            objectInfo.setRelationshipSourceIds(null);
            objectInfo.setRelationshipTargetIds(null);
            objectInfo.setRenditionInfos(null);
            objectInfo.setSupportsDescendants(false);
            objectInfo.setSupportsFolderTree(false);
            objectInfo.setSupportsPolicies(false);
            objectInfo.setSupportsRelationships(false);
            objectInfo.setWorkingCopyId(null);
            objectInfo.setWorkingCopyOriginalId(null);
        }
        else if (ametysObject != null && ametysObject instanceof ModifiableResourceCollection)
        {
            isFolder = true;
            typeId = BaseTypeId.CMIS_FOLDER.value();
            objectInfo.setBaseType(BaseTypeId.CMIS_FOLDER);
            objectInfo.setTypeId(typeId);
            objectInfo.setContentType(null);
            objectInfo.setFileName(null);
            objectInfo.setHasAcl(false);
            objectInfo.setHasContent(false);
            objectInfo.setVersionSeriesId(null);
            objectInfo.setIsCurrentVersion(true);
            objectInfo.setRelationshipSourceIds(null);
            objectInfo.setRelationshipTargetIds(null);
            objectInfo.setRenditionInfos(null);
            objectInfo.setSupportsDescendants(true);
            objectInfo.setSupportsFolderTree(true);
            objectInfo.setSupportsPolicies(false);
            objectInfo.setSupportsRelationships(false);
            objectInfo.setWorkingCopyId(null);
            objectInfo.setWorkingCopyOriginalId(null);
        }
        else
        {
            throw new IllegalArgumentException("Resource not found");
        }
        
        try 
        {
            PropertiesImpl result = new PropertiesImpl();

            addPropertyId(result, typeId, filter, PropertyIds.OBJECT_ID, ametysObject.getId(), factory);
            objectInfo.setId(ametysObject.getId());

            // name
            String name = ametysObject.getName();
            addPropertyString(result, typeId, filter, PropertyIds.NAME, name, factory);
            objectInfo.setName(name);

            if (isFolder)
            {
                ModifiableResourceCollection folder = (ModifiableResourceCollection) ametysObject;
                addPropertyString(result, typeId, filter, PropertyIds.CREATED_BY, USER_UNKNOWN, factory);
                addPropertyString(result, typeId, filter, PropertyIds.LAST_MODIFIED_BY, USER_UNKNOWN, factory);
                objectInfo.setCreatedBy(USER_UNKNOWN);
                addPropertyString(result, typeId, filter, PropertyIds.DESCRIPTION, folder.getDescription(), factory);
                
                // base type and type name
                addPropertyId(result, typeId, filter, PropertyIds.BASE_TYPE_ID, BaseTypeId.CMIS_FOLDER.value(), factory);
                addPropertyId(result, typeId, filter, PropertyIds.OBJECT_TYPE_ID, BaseTypeId.CMIS_FOLDER.value(), factory);
                
                String path = ((ModifiableResourceCollection) ametysObject).getExplorerPath();
                ModifiableResourceCollection root = getRoot(project, factory);
                path = path.substring(root.getExplorerPath().length());
                if (path.length() == 0)
                {
                    path = "/";
                }
                addPropertyString(result, typeId, filter, PropertyIds.PATH, path, factory);

                // folder properties
                
                //On évite d'appeler isRoot qui ferait un appel JCR de plus
                if (ametysObject.equals(root))
                {
                    addPropertyId(result, typeId, filter, PropertyIds.PARENT_ID, null, factory);
                    objectInfo.setHasParent(false);
                }
                else
                {
                    String parentId = ametysObject.getParent().getId();
                    addPropertyId(result, typeId, filter, PropertyIds.PARENT_ID, parentId, factory);
                    objectInfo.setHasParent(true);
                }

                addPropertyIdList(result, typeId, filter, PropertyIds.ALLOWED_CHILD_OBJECT_TYPE_IDS, null, factory);
            }
            else
            {
                ModifiableResource modifiableResource = (ModifiableResource) ametysObject;
                String author = modifiableResource.getCreator().getLogin();
                String contributor = modifiableResource.getLastContributor().getLogin();
                addPropertyString(result, typeId, filter, PropertyIds.CREATED_BY, author, factory);
                addPropertyString(result, typeId, filter, PropertyIds.LAST_MODIFIED_BY, contributor, factory);
                addPropertyString(result, typeId, filter, PropertyIds.DESCRIPTION, modifiableResource.getDCDescription(), factory);
                objectInfo.setCreatedBy(author);
                
                Date lastModifiedDate = modifiableResource.getLastModified();
                GregorianCalendar lastModified = new GregorianCalendar();
                lastModified.setTime(lastModifiedDate);
                Date creationDate = modifiableResource.getCreationDate();
                GregorianCalendar creation = new GregorianCalendar();
                creation.setTime(creationDate);
                addPropertyDateTime(result, typeId, filter, PropertyIds.CREATION_DATE, creation, factory);
                addPropertyDateTime(result, typeId, filter, PropertyIds.LAST_MODIFICATION_DATE, lastModified, factory);
                objectInfo.setCreationDate(creation);
                objectInfo.setLastModificationDate(lastModified);
                
             // base type and type name
                addPropertyId(result, typeId, filter, PropertyIds.BASE_TYPE_ID, BaseTypeId.CMIS_DOCUMENT.value(), factory);
                addPropertyId(result, typeId, filter, PropertyIds.OBJECT_TYPE_ID, BaseTypeId.CMIS_DOCUMENT.value(), factory);

                // file properties
                addPropertyBoolean(result, typeId, filter, PropertyIds.IS_IMMUTABLE, false, factory);
                addPropertyBoolean(result, typeId, filter, PropertyIds.IS_LATEST_VERSION, true, factory);
                addPropertyBoolean(result, typeId, filter, PropertyIds.IS_MAJOR_VERSION, true, factory);
                addPropertyBoolean(result, typeId, filter, PropertyIds.IS_LATEST_MAJOR_VERSION, true, factory);
                addPropertyString(result, typeId, filter, PropertyIds.VERSION_LABEL, modifiableResource.getName(), factory);
                //addPropertyId(result, typeId, filter, PropertyIds.VERSION_SERIES_ID, fileToId(file), factory);
                addPropertyBoolean(result, typeId, filter, PropertyIds.IS_VERSION_SERIES_CHECKED_OUT, false, factory);
                addPropertyString(result, typeId, filter, PropertyIds.VERSION_SERIES_CHECKED_OUT_BY, null, factory);
                addPropertyString(result, typeId, filter, PropertyIds.VERSION_SERIES_CHECKED_OUT_ID, null, factory);
                addPropertyString(result, typeId, filter, PropertyIds.CHECKIN_COMMENT, "", factory);
                if (context.getCmisVersion() != CmisVersion.CMIS_1_0) 
                {
                    addPropertyBoolean(result, typeId, filter, PropertyIds.IS_PRIVATE_WORKING_COPY, false, factory);
                }

                
                if (modifiableResource.getLength() == 0) 
                {
                    addPropertyBigInteger(result, typeId, filter, PropertyIds.CONTENT_STREAM_LENGTH, null, factory);
                    addPropertyString(result, typeId, filter, PropertyIds.CONTENT_STREAM_MIME_TYPE, null, factory);
                    addPropertyString(result, typeId, filter, PropertyIds.CONTENT_STREAM_FILE_NAME, null, factory);

                    objectInfo.setHasContent(false);
                    objectInfo.setContentType(null);
                    objectInfo.setFileName(null);
                }
                else 
                {
                    
                    addPropertyInteger(result, typeId, filter, PropertyIds.CONTENT_STREAM_LENGTH, modifiableResource.getLength(), factory);
                    addPropertyString(result, typeId, filter, PropertyIds.CONTENT_STREAM_MIME_TYPE,
                            modifiableResource.getMimeType(), factory);
                    addPropertyString(result, typeId, filter, PropertyIds.CONTENT_STREAM_FILE_NAME, modifiableResource.getName(), factory);

                    objectInfo.setHasContent(true);
                    objectInfo.setContentType(modifiableResource.getMimeType());
                    objectInfo.setFileName(modifiableResource.getName());
                }

                addPropertyId(result, typeId, filter, PropertyIds.CONTENT_STREAM_ID, null, factory);
            }

            

            // change token - always null
            addPropertyString(result, typeId, filter, PropertyIds.CHANGE_TOKEN, null, factory);

            // CMIS 1.1 properties
            if (context.getCmisVersion() != CmisVersion.CMIS_1_0)
            {
                addPropertyString(result, typeId, filter, PropertyIds.DESCRIPTION, null, factory);
                addPropertyIdList(result, typeId, filter, PropertyIds.SECONDARY_OBJECT_TYPE_IDS, null, factory);
            }

            return result;
        }
        catch (CmisBaseException cbe)
        {
            throw cbe;
        }
        catch (Exception e)
        {
            throw new CmisRuntimeException(e.getMessage(), e);
        }
    }
    
    /**
     * get children of a folder
     * @param context call context
     * @param folderId folder Id
     * @param project Project
     * @param filter filters
     * @param includeAllowableActions allowable actions
     * @param includePathSegment include path segment
     * @param maxItems max items
     * @param skipCount skip count
     * @param objectInfos object infos
     * @param factory factory
     * @return ObjectInFolderList
     */
    public ObjectInFolderList getChildren(CallContext context, String folderId, Project project, String filter,
            Boolean includeAllowableActions, Boolean includePathSegment, BigInteger maxItems, BigInteger skipCount,
            ObjectInfoHandler objectInfos, CmisServiceFactory factory)
    {
        boolean userReadOnly = false;
        
        // split filter
        Set<String> filterCollection = CmisUtils.splitFilter(filter);

        // set defaults if values not set
        boolean iaa = CmisUtils.getBooleanParameter(includeAllowableActions, false);
        boolean ips = CmisUtils.getBooleanParameter(includePathSegment, false);

        // skip and max
        int skip = skipCount == null ? 0 : skipCount.intValue();
        if (skip < 0)
        {
            skip = 0;
        }

        int max = maxItems == null ? Integer.MAX_VALUE : maxItems.intValue();
        if (max < 0)
        {
            max = Integer.MAX_VALUE;
        }

        // get the folder
        AmetysObject ametysObject = factory.getResolver().resolveById(folderId);
        if (!(ametysObject instanceof ModifiableResourceCollection)) 
        {
            throw new CmisObjectNotFoundException("Not a folder!");
        }
        ModifiableResourceCollection folder = (ModifiableResourceCollection) ametysObject;

        // set object info of the the folder
        if (context.isObjectInfoRequired())
        {
            compileObjectData(context, ametysObject, project, null, false, false, userReadOnly, objectInfos, factory);
        }

        // prepare result
        ObjectInFolderListImpl result = new ObjectInFolderListImpl();
        result.setObjects(new ArrayList<>());
        result.setHasMoreItems(false);
        int count = 0;

        // iterate through children
        AmetysObjectIterable<AmetysObject> children = folder.getChildren();
        for (AmetysObject child : children)
        //for (File child : folder.listFiles())
        {
            // skip nodes other than ModifiableResource or ModifiableResourceCollection
            if (!(child instanceof ModifiableResource || child instanceof ModifiableResourceCollection))
            {
                continue;
            }

            count++;

            if (skip > 0)
            {
                skip--;
                continue;
            }

            if (result.getObjects().size() >= max)
            {
                result.setHasMoreItems(true);
                continue;
            }

            // build and add child object
            ObjectInFolderDataImpl objectInFolder = new ObjectInFolderDataImpl();
            objectInFolder.setObject(compileObjectData(context, child, project, filterCollection, iaa, false, userReadOnly,
                    objectInfos, factory));
            if (ips)
            {
                objectInFolder.setPathSegment(child.getName());
            }

            result.getObjects().add(objectInFolder);
        }

        result.setNumItems(BigInteger.valueOf(count));

        return result;
    }
    
    /**
     * get the parents of an object
     * @param context call context
     * @param objectId Object Id
     * @param project Project
     * @param filter filters
     * @param includeAllowableActions allowable actions
     * @param includeRelativePathSegment relative path segment
     * @param objectInfos object infos
     * @param factory factory 
     * @return List of ObjectParentData
     */
    public List<ObjectParentData> getObjectParents(CallContext context, String objectId, Project project, String filter,
            Boolean includeAllowableActions, Boolean includeRelativePathSegment, ObjectInfoHandler objectInfos, CmisServiceFactory factory)
    {
        boolean userReadOnly = false;

        // split filter
        Set<String> filterCollection = CmisUtils.splitFilter(filter);

        // set defaults if values not set
        boolean iaa = CmisUtils.getBooleanParameter(includeAllowableActions, false);
        boolean irps = CmisUtils.getBooleanParameter(includeRelativePathSegment, false);

        // get the file or folder
        //File file = getFile(objectId);

        // don't climb above the root folder
        AmetysObject ametysObject = factory.getResolver().resolveById(objectId);
        if (isRoot(ametysObject, project, factory))
        {
            return Collections.emptyList();
        }

        // set object info of the the object
        if (context.isObjectInfoRequired())
        {
            compileObjectData(context, ametysObject, project, null, false, false, userReadOnly, objectInfos, factory);
        }

        // get parent folder
        AmetysObject parent = ametysObject.getParent();
        //File parent = file.getParentFile();
        ObjectData object = compileObjectData(context, parent, project, filterCollection, iaa, false, userReadOnly, objectInfos, factory);

        ObjectParentDataImpl result = new ObjectParentDataImpl();
        result.setObject(object);
        if (irps)
        {
            result.setRelativePathSegment(ametysObject.getName());
        }

        return Collections.<ObjectParentData> singletonList(result);
    }
    /**
     * get the inputstream to read a file
     * @param context call context
     * @param objectId object Id
     * @param offset offset
     * @param length lenght
     * @param factory factory
     * @return ContentStream
     */
    public ContentStream getContentStream(CallContext context, String objectId, BigInteger offset, BigInteger length, CmisServiceFactory factory)
    {
        // get the file
        AmetysObject ametysObject = factory.getResolver().resolveById(objectId);
        if (!(ametysObject instanceof ModifiableResource))
        {
            throw new CmisStreamNotSupportedException("Not a file!");
        }
        ModifiableResource res = (ModifiableResource) ametysObject;
        
        if (res.getLength() == 0)
        {
            throw new CmisConstraintException("Document has no content!");
        }

        InputStream stream = null;
        stream = res.getInputStream();
        if (offset != null || length != null)
        {
            stream = new CmisContentRangeInputStream(stream, offset, length);
        }

        // compile data
        ContentStreamImpl result;
        if (offset != null && offset.longValue() > 0
            || length != null)
        {
            result = new PartialContentStreamImpl();
        }
        else
        {
            result = new ContentStreamImpl();
        }

        result.setFileName(res.getName());
        result.setLength(BigInteger.valueOf(res.getLength()));
        result.setMimeType(res.getMimeType());
        result.setStream(stream);

        return result;
    }

    /**
     * get an object by it's path
     * @param context call context
     * @param project Project
     * @param folderPath path of the object/folder
     * @param filter filters for metadata
     * @param includeAllowableActions allowable actions
     * @param includeACL ACL
     * @param objectInfos infos
     * @param factory factory
     * @return datas of the object
     */
    public ObjectData getObjectByPath(CallContext context, Project project, String folderPath, String filter,
            boolean includeAllowableActions, boolean includeACL, ObjectInfoHandler objectInfos, CmisServiceFactory factory)
    {
        boolean userReadOnly = false;

        // split filter
        Set<String> filterCollection = CmisUtils.splitFilter(filter);

        // check path
        if (folderPath == null || folderPath.length() == 0 || folderPath.charAt(0) != '/')
        {
            throw new CmisInvalidArgumentException("Invalid folder path!");
        }

        // get the file or folder
        AmetysObject root = getRoot(project, factory);
        AmetysObject file = null;
        if (folderPath.length() == 1)
        {
            file = root;
        }
        else
        {
            file = factory.getResolver().resolveByPath(root.getPath() + folderPath);
            //String path = folderPath.replace('/', File.separatorChar).substring(1);
            //file = new File(root, path);
        }

        if (file == null)
        {
            throw new CmisObjectNotFoundException("Path doesn't exist.");
        }
        return compileObjectData(context, file, project, filterCollection, includeAllowableActions, includeACL, userReadOnly,
                objectInfos, factory);
    }
    
    /**
     * Create a new folder in another one
     * @param context call context
     * @param properties properties
     * @param project Project
     * @param folderId folder Id
     * @param factory factory
     * @return id of the created folder
     */
    public String createFolder(CallContext context, Properties properties, Project project, String folderId, CmisServiceFactory factory)
    {
        // check properties
        checkNewProperties(properties, BaseTypeId.CMIS_FOLDER, factory);

        // get parent File
        AmetysObject ametysObject = factory.getResolver().resolveById(folderId);
        if (!(ametysObject instanceof ModifiableResourceCollection))
        {
            throw new CmisObjectNotFoundException("Parent is not a folder!");
        }
        
        // create the folder
        String name = CmisUtils.getStringProperty(properties, PropertyIds.NAME);
        String description = CmisUtils.getStringProperty(properties, PropertyIds.DESCRIPTION, "");
        
        Map<String, Object> addFolder;
        addFolder = factory.getWorkspaceExplorerResourceDAO().addFolder(folderId, name, description, true);
        return (String) addFolder.get("id");
    }
    
    /**
     * Create a new document in a folder
     * @param context call context
     * @param properties properties
     * @param project Project
     * @param folderId folder Id
     * @param contentStream content Stream
     * @param versioningState versionning State
     * @param factory factory
     * @return id of the created document
     */
    public String createDocument(CallContext context, Properties properties, Project project, String folderId,
            ContentStream contentStream, VersioningState versioningState, CmisServiceFactory factory)
    {
        // check versioning state
        if (VersioningState.NONE != versioningState)
        {
            throw new CmisConstraintException("Versioning not supported!");
        }

        // check properties
        checkNewProperties(properties, BaseTypeId.CMIS_DOCUMENT, factory);

        String name = CmisUtils.getStringProperty(properties, PropertyIds.NAME);
        ResourceCollection collection = factory.getResolver().resolveById(folderId);
        if (!(collection instanceof ModifiableResourceCollection))
        {
            throw new IllegalArgumentException("Cannot create file on a non modifiable folder '" + folderId + "'");
        }
        
        ModifiableResourceCollection mCollection = (ModifiableResourceCollection) collection;
        factory.getAddOrUpdateResourceHelper().checkAddResourceRight(mCollection);
        ResourceOperationResult resourceOperation = factory.getAddOrUpdateResourceHelper().performResourceOperation(contentStream.getStream(), name, mCollection, ResourceOperationMode.ADD_RENAME);
        if (!resourceOperation.isSuccess())
        {
            throw new CmisStorageException("Could not create file: " + resourceOperation.getErrorMessage());
        }

        String id = resourceOperation.getResource().getId();
        return id;
    }
    
    /**
     * CMIS deleteObject.
     * @param context call context
     * @param objectId object ID
     * @param factory factory
     */
    public void deleteObject(CallContext context, String objectId, CmisServiceFactory factory)
    {
        List<String> toDelete = new ArrayList<>(1);
        toDelete.add(objectId);
        Map deleted = factory.getWorkspaceExplorerResourceDAO().deleteObject(toDelete);
        
        if (deleted.containsKey("message"))
        {
            throw new CmisStorageException("Deletion failed : " + deleted.get("message"));
        }
    }
    
    /**
     * CMIS getFolderParent.
     * @param context call context
     * @param project Project
     * @param folderId folder Id
     * @param filter filters
     * @param objectInfos objectInfos
     * @param factory factory
     * @return ObjectData
     */
    public ObjectData getFolderParent(CallContext context, Project project, String folderId, String filter, ObjectInfoHandler objectInfos, CmisServiceFactory factory)
    {
        boolean userReadOnly = false;
        AmetysObject ametysObject = factory.getResolver().resolveById(folderId);
        if (isRoot(ametysObject, project, factory))
        {
            throw new CmisInvalidArgumentException("The root folder has no parent!");
        }
        AmetysObject parent = ametysObject.getParent();
        
        Set<String> filterCollection = CmisUtils.splitFilter(filter);
        ObjectData object = compileObjectData(context, parent, project, filterCollection, false, false, userReadOnly, objectInfos, factory);
        
        return object;
    }
    
    /**
     * CMIS deleteTree.
     * @param context call context
     * @param project Project
     * @param folderId folder Id
     * @param factory factory
     * @return FailedToDeleteData
     */
    public FailedToDeleteData deleteTree(CallContext context, Project project, String folderId, CmisServiceFactory factory)
    {
        AmetysObject ametysObject = factory.getResolver().resolveById(folderId);
        if (ametysObject instanceof ModifiableResourceCollection)
        {
            ModifiableResourceCollection folder = (ModifiableResourceCollection) ametysObject;
            AmetysObjectIterable<AmetysObject> children = folder.getChildren();
            List<String> toDelete = new ArrayList<>();
            for (AmetysObject child : children)
            {
                toDelete.add(child.getId());
                
            }
            Map deletedMsg = factory.getWorkspaceExplorerResourceDAO().deleteObject(toDelete);
            
            if (deletedMsg.containsKey("message"))
            {
                throw new CmisStorageException("Deletion failed : " + deletedMsg.get("message"));
            }
        }
        else
        {
            throw new CmisConstraintException("Object is not a folder!");
        }
        
        FailedToDeleteDataImpl result = new FailedToDeleteDataImpl();
        result.setIds(new ArrayList<>());

        return result;
    }
    
    /**
     * CMIS updateProperties.
     * @param context call context
     * @param project Project
     * @param objectIdHolder objectId in a holder
     * @param properties properties
     * @param objectInfos object infos
     * @param factory factory
     * @return ObjectData
     */
    public ObjectData updateProperties(CallContext context, Project project, Holder<String> objectIdHolder, Properties properties,
            ObjectInfoHandler objectInfos, CmisServiceFactory factory)
    {
        // check object id
        if (objectIdHolder == null || objectIdHolder.getValue() == null)
        {
            throw new CmisInvalidArgumentException("Id is not valid!");
        }
        String objectId = objectIdHolder.getValue();
        boolean userReadOnly = false;
        
        AmetysObject ametysObject = factory.getResolver().resolveById(objectId);

        if (ametysObject != null && ametysObject instanceof ModifiableResource)
        {
            ModifiableResource resource = (ModifiableResource) ametysObject;
            String inputName = CmisUtils.getStringProperty(properties, PropertyIds.NAME, resource.getName());
            String description = CmisUtils.getStringProperty(properties, PropertyIds.DESCRIPTION, resource.getDCDescription());
            Collection<String> tags =  new ArrayList<>(Arrays.asList(resource.getKeywords()));
            
            factory.getWorkspaceExplorerResourceDAO().editFile(objectId, inputName, description, tags);
            
        }
        else if (ametysObject != null && ametysObject instanceof ModifiableResourceCollection)
        {
            ModifiableResourceCollection folder = (ModifiableResourceCollection) ametysObject;
            String inputName = CmisUtils.getStringProperty(properties, PropertyIds.NAME, folder.getName());
            String description = CmisUtils.getStringProperty(properties, PropertyIds.DESCRIPTION, folder.getDescription());
            if (description == null)
            {
                description = StringUtils.EMPTY;
            }
            factory.getWorkspaceExplorerResourceDAO().editFolder(objectId, inputName, description);
        }
        else
        {
            throw new CmisObjectNotFoundException("File not found!");
        }
        
        AmetysObject modified = factory.getResolver().resolveById(objectId);
        return compileObjectData(context, modified, project, null, false, false, userReadOnly, objectInfos, factory);
    }
    
    /**
     * CMIS setContentStream, deleteContentStream, and appendContentStream.
     * @param context call context
     * @param project Project
     * @param objectIdHolder object Id in a holder
     * @param overwriteFlag overwrite
     * @param contentStream inputStream
     * @param append append
     * @param factory factory
     */
    public void changeContentStream(CallContext context, Project project, Holder<String> objectIdHolder, Boolean overwriteFlag,
            ContentStream contentStream, boolean append, CmisServiceFactory factory)
    {
        if (objectIdHolder == null || objectIdHolder.getValue() == null)
        {
            throw new CmisInvalidArgumentException("Id is not valid!");
        }
        String objectId = objectIdHolder.getValue();
        
        AmetysObject ametysObject = factory.getResolver().resolveById(objectId);
        
     // check overwrite
        boolean owf = CmisUtils.getBooleanParameter(overwriteFlag, true);
        if (!owf && contentStream.getLength() > 0)
        {
            throw new CmisContentAlreadyExistsException("Content already exists!");
        }
        
        if (ametysObject instanceof ModifiableResource)
        {
            factory.getAddOrUpdateResourceHelper().checkAddResourceRight((ModifiableResourceCollection) ametysObject);
            
            String name = ametysObject.getName();
            ResourceOperationResult resourceOperation = factory.getAddOrUpdateResourceHelper().performResourceOperation(contentStream.getStream(), name, ametysObject.getParent(), ResourceOperationMode.UPDATE);
                
            if (!resourceOperation.isSuccess())
            {
                throw new CmisRuntimeException("Impossible to update file : " + resourceOperation.getErrorMessage());
            }
        }
        else
        {
            throw new CmisObjectNotFoundException("File not found!");
        }
    }
    /**
     * CMIS moveObject.
     * @param context call context
     * @param project Project
     * @param objectId objectId in a holder
     * @param targetFolderId folderId
     * @param objectInfos objectInfos
     * @param factory factory
     * @return ObjectData
     */
    public ObjectData moveObject(CallContext context, Project project, Holder<String> objectId, String targetFolderId,
            ObjectInfoHandler objectInfos, CmisServiceFactory factory)
    {
        boolean userReadOnly = false;

        if (objectId == null)
        {
            throw new CmisInvalidArgumentException("Id is not valid!");
        }

        List<String> documentIds = new ArrayList<>(1);
        documentIds.add(objectId.getValue());
        
        Map<String, Object> result;
        try
        {
            result = factory.getWorkspaceExplorerResourceDAO().moveDocuments(documentIds, targetFolderId);
            if (result.containsKey("message"))
            {
                throw new CmisStorageException((String) result.get("message"));
            }
            else
            {
                AmetysObject ametysObject = factory.getResolver().resolveById(objectId.getValue());
                return compileObjectData(context, ametysObject, project, null, false, false, userReadOnly, objectInfos, factory);
            }
        }
        catch (RepositoryException e)
        {
            throw new CmisStorageException("Impossible to move file", e);
        }
        
        
    }
    
    /*
     * private helpers
     * 
     */
    
    /*
     * Checks a property set for a new object.
     */
    private void checkNewProperties(Properties properties, BaseTypeId baseTypeId, CmisServiceFactory factory)
    {
        // check properties
        if (properties == null || properties.getProperties() == null)
        {
            throw new CmisInvalidArgumentException("Properties must be set!");
        }

        // check the name
        /*
        String name = CmisUtils.getStringProperty(properties, PropertyIds.NAME);
        if (!isValidName(name)) {
            throw new CmisNameConstraintViolationException("Name is not valid!");
        }*/

        // check the type
        String typeId = CmisUtils.getObjectTypeId(properties);
        if (typeId == null)
        {
            throw new CmisInvalidArgumentException("Type Id is not set!");
        }

        TypeDefinition type = factory.getTypeManager().getInternalTypeDefinition(typeId);
        if (type == null)
        {
            throw new CmisObjectNotFoundException("Type '" + typeId + "' is unknown!");
        }

        if (type.getBaseTypeId() != baseTypeId)
        {
            if (baseTypeId == BaseTypeId.CMIS_DOCUMENT)
            {
                throw new CmisInvalidArgumentException("Type is not a document type!");
            }
            else if (baseTypeId == BaseTypeId.CMIS_DOCUMENT)
            {
                throw new CmisInvalidArgumentException("Type is not a folder type!");
            }
            else
            {
                throw new CmisRuntimeException("A file system does not support a " + baseTypeId.value() + " type!");
            }
        }

        // check type properties
        checkTypeProperties(properties, typeId, factory, true);

        // check if required properties are missing
        for (PropertyDefinition<?> propDef : type.getPropertyDefinitions().values())
        {
            if (propDef.isRequired() && !properties.getProperties().containsKey(propDef.getId())
                    && propDef.getUpdatability() != Updatability.READONLY)
            {
                throw new CmisConstraintException("Property '" + propDef.getId() + "' is required!");
            }
        }
    }
    
    /*
     * Checks if the property belong to the type and are settable.
     */
    private void checkTypeProperties(Properties properties, String typeId, CmisServiceFactory factory, boolean isCreate)
    {
        // check type
        TypeDefinition type = factory.getTypeManager().getInternalTypeDefinition(typeId);
        if (type == null)
        {
            throw new CmisObjectNotFoundException("Type '" + typeId + "' is unknown!");
        }

        // check if all required properties are there
        for (PropertyData<?> prop : properties.getProperties().values())
        {
            PropertyDefinition<?> propType = type.getPropertyDefinitions().get(prop.getId());

            // do we know that property?
            if (propType == null)
            {
                throw new CmisConstraintException("Property '" + prop.getId() + "' is unknown!");
            }

            // can it be set?
            if (propType.getUpdatability() == Updatability.READONLY)
            {
                throw new CmisConstraintException("Property '" + prop.getId() + "' is readonly!");
            }

            if (!isCreate)
            {
                // can it be set?
                if (propType.getUpdatability() == Updatability.ONCREATE)
                {
                    throw new CmisConstraintException("Property '" + prop.getId() + "' cannot be updated!");
                }
            }
        }
    }
    
    private void addPropertyId(PropertiesImpl props, String typeId, Set<String> filter, String id, String value, CmisServiceFactory factory)
    {
        if (!checkAddProperty(props, typeId, filter, id, factory))
        {
            return;
        }

        props.addProperty(new PropertyIdImpl(id, value));
    }

    private void addPropertyIdList(PropertiesImpl props, String typeId, Set<String> filter, String id,
            List<String> value, CmisServiceFactory factory)
    {
        if (!checkAddProperty(props, typeId, filter, id, factory))
        {
            return;
        }

        props.addProperty(new PropertyIdImpl(id, value));
    }

    private void addPropertyString(PropertiesImpl props, String typeId, Set<String> filter, String id, String value, CmisServiceFactory factory)
    {
        if (!checkAddProperty(props, typeId, filter, id, factory))
        {
            return;
        }

        props.addProperty(new PropertyStringImpl(id, value));
    }

    private void addPropertyInteger(PropertiesImpl props, String typeId, Set<String> filter, String id, long value, CmisServiceFactory factory)
    {
        addPropertyBigInteger(props, typeId, filter, id, BigInteger.valueOf(value), factory);
    }

    private void addPropertyBigInteger(PropertiesImpl props, String typeId, Set<String> filter, String id,
            BigInteger value, CmisServiceFactory factory)
    {
        if (!checkAddProperty(props, typeId, filter, id, factory))
        {
            return;
        }

        props.addProperty(new PropertyIntegerImpl(id, value));
    }

    private void addPropertyBoolean(PropertiesImpl props, String typeId, Set<String> filter, String id, boolean value, CmisServiceFactory factory)
    {
        if (!checkAddProperty(props, typeId, filter, id, factory))
        {
            return;
        }

        props.addProperty(new PropertyBooleanImpl(id, value));
    }

    private void addPropertyDateTime(PropertiesImpl props, String typeId, Set<String> filter, String id,
            GregorianCalendar value, CmisServiceFactory factory)
    {
        if (!checkAddProperty(props, typeId, filter, id, factory))
        {
            return;
        }

        props.addProperty(new PropertyDateTimeImpl(id, value));
    }

    private boolean checkAddProperty(Properties properties, String typeId, Set<String> filter, String id, CmisServiceFactory factory)
    {
        if (properties == null || properties.getProperties() == null)
        {
            throw new IllegalArgumentException("Properties must not be null!");
        }

        if (id == null)
        {
            throw new IllegalArgumentException("Id must not be null!");
        }

        TypeDefinition type = factory.getTypeManager().getInternalTypeDefinition(typeId);
        if (type == null)
        {
            throw new IllegalArgumentException("Unknown type: " + typeId);
        }
        if (!type.getPropertyDefinitions().containsKey(id))
        {
            throw new IllegalArgumentException("Unknown property: " + id);
        }

        String queryName = type.getPropertyDefinitions().get(id).getQueryName();

        if (queryName != null && filter != null)
        {
            if (!filter.contains(queryName))
            {
                return false;
            }
            else
            {
                filter.remove(queryName);
            }
        }

        return true;
    }
    
    /*
     * Compiles the allowable actions for a file or folder.
     */
    private AllowableActions compileAllowableActions(AmetysObject file, Project project, CmisServiceFactory factory, boolean userReadOnly)
    {
        if (file == null)
        {
            throw new IllegalArgumentException("File must not be null!");
        }

        boolean isReadOnly = false;
        boolean isRoot = isRoot(file, project, factory);

        Set<Action> aas = EnumSet.noneOf(Action.class);

        addAction(aas, Action.CAN_GET_OBJECT_PARENTS, !isRoot);
        addAction(aas, Action.CAN_GET_PROPERTIES, true);
        addAction(aas, Action.CAN_UPDATE_PROPERTIES, !userReadOnly && !isReadOnly);
        addAction(aas, Action.CAN_MOVE_OBJECT, !userReadOnly && !isRoot);
        addAction(aas, Action.CAN_DELETE_OBJECT, !userReadOnly && !isReadOnly && !isRoot);
        addAction(aas, Action.CAN_GET_ACL, true);

        if (file instanceof ModifiableResourceCollection)
        {
            addAction(aas, Action.CAN_GET_DESCENDANTS, true);
            addAction(aas, Action.CAN_GET_CHILDREN, true);
            addAction(aas, Action.CAN_GET_FOLDER_PARENT, !isRoot);
            addAction(aas, Action.CAN_GET_FOLDER_TREE, true);
            addAction(aas, Action.CAN_CREATE_DOCUMENT, !userReadOnly);
            addAction(aas, Action.CAN_CREATE_FOLDER, !userReadOnly);
            addAction(aas, Action.CAN_DELETE_TREE, !userReadOnly && !isReadOnly);
        }
        else if (file instanceof ModifiableResource)
        {
            ModifiableResource res = (ModifiableResource) file;
            addAction(aas, Action.CAN_GET_CONTENT_STREAM, res.getLength() > 0);
            addAction(aas, Action.CAN_SET_CONTENT_STREAM, !userReadOnly && !isReadOnly);
            addAction(aas, Action.CAN_DELETE_CONTENT_STREAM, !userReadOnly && !isReadOnly);
            addAction(aas, Action.CAN_GET_ALL_VERSIONS, true);
        }

        AllowableActionsImpl result = new AllowableActionsImpl();
        result.setAllowableActions(aas);

        return result;
    }
    private void addAction(Set<Action> aas, Action action, boolean condition)
    {
        if (condition)
        {
            aas.add(action);
        }
    }
    
    private Boolean isRoot(ModifiableResourceCollection object, Project project, DocumentWorkspaceModule documentModule)
    {
        ModifiableResourceCollection documentRoot = documentModule.getModuleRoot(project, false);
        return documentRoot.getId().equals(object.getId());
    }
    private Boolean isRoot(AmetysObject ametysObject, Project project, CmisServiceFactory factory)
    {
        if (ametysObject instanceof ModifiableResourceCollection)
        {
            return isRoot((ModifiableResourceCollection) ametysObject, project, factory.getDocumentModule());
        }
        else
        {
            return false;
        }
    }
    
    private ModifiableResourceCollection getRoot(Project project, CmisServiceFactory factory)
    {
        return factory.getDocumentModule().getModuleRoot(project, false);
    }
    
    
}
