/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.cmis;

import java.util.HashSet;
import java.util.Set;

import org.apache.chemistry.opencmis.commons.PropertyIds;
import org.apache.chemistry.opencmis.commons.data.Properties;
import org.apache.chemistry.opencmis.commons.data.PropertyData;
import org.apache.chemistry.opencmis.commons.data.PropertyId;
import org.apache.chemistry.opencmis.commons.data.PropertyString;
import org.apache.chemistry.opencmis.commons.exceptions.CmisInvalidArgumentException;

/**
 * utils class for CMIS server
 *
 */
public final class CmisUtils
{

    private CmisUtils()
    {
        //nothing here
    }
    /**
     * Returns the boolean value of the given value or the default value if the
     * given value is <code>null</code>.
     * @param value value
     * @param def default
     * @return boolean
     */
    public static boolean getBooleanParameter(Boolean value, boolean def)
    {
        if (value == null)
        {
            return def;
        }

        return value.booleanValue();
    }
    /**
     * Returns the first value of a string property.
     * @param properties property
     * @param name name
     * @return String
     */
    public static String getStringProperty(Properties properties, String name)
    {
        return getStringProperty(properties, name, null);
    }
    /**
     * get String property
     * @param properties property
     * @param name name
     * @param defaultValue default value
     * @return String
     */
    public static String getStringProperty(Properties properties, String name, String defaultValue)
    {
        PropertyData<?> property = properties.getProperties().get(name);
        if (!(property instanceof PropertyString))
        {
            return defaultValue;
        }

        return ((PropertyString) property).getFirstValue();
    }
    
    /**
     * Gets the type id from a set of properties.
     * @param properties properties
     * @return type id
     */
    public static String getObjectTypeId(Properties properties)
    {
        PropertyData<?> typeProperty = properties.getProperties().get(PropertyIds.OBJECT_TYPE_ID);
        if (!(typeProperty instanceof PropertyId))
        {
            throw new CmisInvalidArgumentException("Type Id must be set!");
        }

        String typeId = ((PropertyId) typeProperty).getFirstValue();
        if (typeId == null)
        {
            throw new CmisInvalidArgumentException("Type Id must be set!");
        }

        return typeId;
    }

    /**
     * Splits a filter statement into a collection of properties. If
     * <code>filter</code> is <code>null</code>, empty or one of the properties
     * is '*' , an empty collection will be returned.
     * @param filter input filters
     * @return set of filters, or null if no filters
     */
    public static Set<String> splitFilter(String filter)
    {
        if (filter == null || filter.trim().length() == 0)
        {
            return null;
        }

        Set<String> result = new HashSet<>();
        for (String s : filter.split(","))
        {
            String trimmed = s.trim();
            if (trimmed.equals("*") || trimmed.equals(org.apache.chemistry.opencmis.commons.impl.Constants.RENDITION_NONE))
            {
                return null;
            }
            else if (trimmed.length() > 0)
            {
                result.add(trimmed);
            }
        }

        // set a few base properties
        // query name == id (for base type properties)
        result.add(PropertyIds.OBJECT_ID);
        result.add(PropertyIds.OBJECT_TYPE_ID);
        result.add(PropertyIds.BASE_TYPE_ID);

        return result;
    }
}
