/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.comments;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.mentions.AbstractNotifyMentionsObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.forum.ForumWorkspaceModule;
import org.ametys.plugins.workspaces.forum.Thread;
import org.ametys.plugins.workspaces.forum.json.ThreadJSONHelper;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebConstants;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;

/**
 * Observer to send mails to mentioned users in content of {@link Thread}
 */
public class NotifyThreadMentionsObserver extends AbstractNotifyMentionsObserver<Thread> implements Contextualizable
{
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The forum module */
    protected ForumWorkspaceModule _forumModule;
    
    /** The project manager */
    protected ProjectManager _projectManager;

    /** The project tags DAO */
    protected ThreadJSONHelper _threadJSONHelper;

    /** The Avalon context */
    protected Context _context;

    /** The Workspaces helper */
    protected WorkspacesHelper _workspaceHelper;
    
    /** The project rights helper */
    protected ProjectRightHelper _projectRightsHelper;

    /** The rendering context handler */
    protected RenderingContextHandler _renderingContextHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _forumModule = moduleManagerEP.getModule(ForumWorkspaceModule.FORUM_MODULE_ID);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _threadJSONHelper = (ThreadJSONHelper) manager.lookup(ThreadJSONHelper.ROLE);
        _workspaceHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
        _projectRightsHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
    }

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Map<String, Object> arguments = event.getArguments();

        Thread thread = (Thread) arguments.get(ObservationConstants.ARGS_THREAD);
        Project project = _projectManager.getParentProject(thread);
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, project.getName());
        super.observe(event, transientVars);
    }
    
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_THREAD_CREATED);
    }

    @Override
    protected I18nizableText _getMailTitle(MentionableObject mentionableObject)
    {
        Thread thread = (Thread) mentionableObject.ametysObject();
        Project project = _projectManager.getParentProject(thread);

        List<String> i18nParams = List.of(project.getTitle());
        
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_WORKSPACES_MENTION_MAIL_SUBJECT", i18nParams);
    }

    @Override
    protected I18nizableText _getMailSubject(MentionableObject mentionableObject)
    {
        Project project = _projectManager.getParentProject(mentionableObject.ametysObject());
        List<String> i18nParams = List.of(project.getTitle());
        return new I18nizableText("plugin.cms", "PLUGINS_CMS_AMETYS_OBJECT_THREAD_MENTION_MAIL_SUBJECT", i18nParams);
    }

    @Override
    protected MentionableObject _getMentionableObjectFromArguments(Map<String, Object> arguments) throws Exception
    {
        Thread thread = (Thread) arguments.get(ObservationConstants.ARGS_THREAD);
        thread = _resolver.resolveById(thread.getId());
        Project project = _projectManager.getParentProject(thread);
        String content = _getContentFromThread(thread);
        String url = _getUrl(thread, project);
        
        return new MentionableObject(
            _userManager.getUser(thread.getAuthor()),
            content,
            _mentionUtils.extractMentionedUsersFromRichText(content),
            thread.getCreationDate(),
            thread,
            new LinkToAmetysObject(url, new I18nizableText("plugin.workspaces", "PROJECT_MAIL_NOTIFICATION_BODY_FORUMTHREAD_BUTTON_TEXT")),
            _workspaceHelper.getLang(project)
        );
    }
    
    private String _getContentFromThread(Thread thread) throws IOException
    {
        Request request = ContextHelper.getRequest(_context);
        String siteName = (String) request.getAttribute("site");
        Project project = _projectManager.getParentProject(thread);
        if (siteName == null)
        {
            request.setAttribute("site", project.getSite().getName());
        }
        return _threadJSONHelper.richTextToRendering(thread.getContent());
    }
    
    private String _getAmetysObjectTitle(Thread thread)
    {
        return thread.getTitle();
    }

    @Override
    protected String _transformSyntaxTextToReadableTextWithColors(String content, UserIdentity recipient)
    {
        return _mentionUtils.transformRichTextToReadableText(content, recipient);
    }

    @Override
    protected I18nizableText _getMailMessageType()
    {
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_THREAD_MENTION_MAIL_DESCRIPTION_TITLE");
    }

    @Override
    protected I18nizableText _getMailMessage(MentionableObject mentionableObject)
    {
        Thread thread = (Thread) mentionableObject.ametysObject();
        Project project = _projectManager.getParentProject(thread);
        User author = mentionableObject.author();
        List<String> i18nParams = List.of(author.getFullName(), _getAmetysObjectTitle(thread), project.getTitle());
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_THREAD_MENTION_MAIL_MESSAGE", i18nParams);
    }
    
    @Override
    protected boolean _canSendMailToMentionedUser(Thread thread, User authorIdentity, UserIdentity mentionedUserIdentity)
    {
        Project project = _projectManager.getParentProject(thread);
        
        return super._canSendMailToMentionedUser(thread, authorIdentity, mentionedUserIdentity)
            && _projectRightsHelper.hasReadAccessOnModule(project, ForumWorkspaceModule.FORUM_MODULE_ID, mentionedUserIdentity);
    }
    
    private String _getUrl(Thread thread, Project project)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute("forceAbsoluteUrl", true);

        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
        _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);

        try
        {
            return _forumModule.getThreadUri(project, thread.getId());
        }
        finally
        {
            _renderingContextHandler.setRenderingContext(currentContext);
        }
    }
}
