/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.dav;

import java.io.IOException;
import java.io.InputStream;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.TimeZone;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.http.HttpResponse;
import org.apache.cocoon.reading.ServiceableReader;
import org.apache.commons.io.IOUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;

/**
 * Reader for Webdav GET, honouring ETag and If-None-Match headers.
 */
public class WebdavResourceReader extends ServiceableReader
{
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        AmetysObject object = (AmetysObject) request.getAttribute("resource");
        
        Response response = ObjectModelHelper.getResponse(objectModel);

        if (object instanceof Resource)
        {
            Resource resource = (Resource) object;
            response.setHeader("Content-Type", resource.getMimeType());
            
            String lastModified = DateTimeFormatter.RFC_1123_DATE_TIME.withZone(TimeZone.getTimeZone("GMT").toZoneId()).format(DateUtils.asZonedDateTime(resource.getLastModified(), ZoneId.systemDefault()));
            response.setHeader("Last-Modified", lastModified);
            response.setHeader("ETag", resource.getLastModified().toString());
            
            String etag = request.getHeader("If-None-Match");
            if (etag != null && etag.equals(resource.getLastModified().toString()))
            {
                ((HttpResponse) response).setStatus(304);
                response.setHeader("Content-Length", "0");
                return;
            }
            
            try (InputStream is = resource.getInputStream())
            {
                response.setHeader("Content-Length", String.valueOf(resource.getLength()));
                IOUtils.copy(is, out);
            }
        }
    }
}
