/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents.observers;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.documents.onlyoffice.OnlyOfficeManager;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * This observer observes event of resources to generate thumbnail if only office is enabled
 */
public class FileThumbnailObserver implements AsyncObserver, Serviceable
{
    //FIXME pass this observer in async when currentUserProvider return not null current user in async observers
    
    /** The only office manager */
    protected OnlyOfficeManager _onlyOfficeManager;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _onlyOfficeManager = (OnlyOfficeManager) manager.lookup(OnlyOfficeManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public int getPriority()
    {
        return MAX_PRIORITY;
    }

    public boolean supports(Event event)
    {
        return _onlyOfficeManager.isOnlyOfficeAvailable()
            && (
                event.getId().equals(ObservationConstants.EVENT_RESOURCE_CREATED) 
             || event.getId().equals(ObservationConstants.EVENT_RESOURCE_UPDATED)
             || event.getId().equals(ObservationConstants.EVENT_RESOURCE_DELETED)
             || event.getId().equals(ObservationConstants.EVENT_COLLECTION_DELETING)
             || event.getId().equals(org.ametys.plugins.workspaces.ObservationConstants.EVENT_PROJECT_DELETED)
            );
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Map<String, Object> args = event.getArguments();
        UserIdentity currentUser = event.getIssuer();
        
        if (event.getId().equals(org.ametys.plugins.workspaces.ObservationConstants.EVENT_PROJECT_DELETED))
        {
            String projectName = (String) event.getArguments().get(org.ametys.plugins.workspaces.ObservationConstants.ARGS_PROJECT_NAME);
            _onlyOfficeManager.deleteProjectThumbnailsInCache(projectName);
        }
        else
        {
            String parentId = (String) args.get(ObservationConstants.ARGS_PARENT_ID);
            AmetysObject ao = _resolver.resolveById(parentId);
            
            Project project = getProject(ao);
            if (project != null)
            {
                String projectName = project.getName();
                if (event.getId().equals(ObservationConstants.EVENT_RESOURCE_CREATED))
                {
                    @SuppressWarnings("unchecked")
                    Map<String, Resource> resources = (Map<String, Resource>) args.get(ObservationConstants.ARGS_RESOURCES);
                    for (String resourceId : resources.keySet())
                    {
                        _onlyOfficeManager.generateThumbnailInCache(projectName, resourceId, currentUser);
                    }
                }
                else if (event.getId().equals(ObservationConstants.EVENT_RESOURCE_UPDATED))
                {
                    String resourceId = (String) args.get(ObservationConstants.ARGS_ID);
                    _onlyOfficeManager.deleteThumbnailInCache(projectName, resourceId);
                    _onlyOfficeManager.generateThumbnailInCache(projectName, resourceId, currentUser);
                }
                else if (event.getId().equals(ObservationConstants.EVENT_RESOURCE_DELETED))
                {
                    String resourceId = (String) args.get(ObservationConstants.ARGS_ID);
                    _onlyOfficeManager.deleteThumbnailInCache(projectName, resourceId);
                }
                else if (event.getId().equals(ObservationConstants.EVENT_COLLECTION_DELETING))
                {
                    String resourceCollectionId = (String) args.get(ObservationConstants.ARGS_ID);
                    ResourceCollection resourceCollection = _resolver.resolveById(resourceCollectionId);
                    _deleteResourcesInCaches(projectName, resourceCollection);
                }
            }
        }
    }
    
    /**
     * Delete resource thumbnail in cache from the a resource collection
     * @param projectName the project name
     * @param resourceCollection the resource collection
     */
    protected void _deleteResourcesInCaches(String projectName, ResourceCollection resourceCollection)
    {
        for (AmetysObject child : resourceCollection.getChildren())
        {
            if (child instanceof ResourceCollection)
            {
                _deleteResourcesInCaches(projectName, (ResourceCollection) child);
            }
            else if (child instanceof Resource)
            {
                _onlyOfficeManager.deleteThumbnailInCache(projectName, child.getId());
            }
        }
    }
    
    /**
     * Get the parent project
     * @param ao The ametys object
     * @return The parent project or <code>null</code> if not found
     */
    protected Project getProject(AmetysObject ao)
    {
        AmetysObject parentAO = ao;
        while (parentAO != null)
        {
            if (parentAO instanceof Project)
            {
                return (Project) parentAO;
            } 
            parentAO = parentAO.getParent();
        }
        
        return null;
    }
}
