/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents.onlyoffice;

import java.io.File;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.URIUtils;

/**
 * Get the file thumbnail
 */
public class GetThumbnailAction extends ServiceableAction implements ThreadSafe
{
    /** The only office manager */
    protected OnlyOfficeManager _onlyOfficeManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _onlyOfficeManager = (OnlyOfficeManager) smanager.lookup(OnlyOfficeManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        String projectName = parameters.getParameter("projectName");
        String resourceId = parameters.getParameter("resourceId");

        try
        {
            Map<String, String> info = _getThumbnailInfo(projectName, resourceId);
            if (info != null)
            {
                return info;
            }
            
            _onlyOfficeManager.generateThumbnailInCache(projectName, resourceId, _currentUserProvider.getUser());
            
            return _getThumbnailInfo(projectName, resourceId);
        }
        catch (Exception e) 
        {
            getLogger().error("An error occurred getting thumbnail for project '" + projectName + "' and resourceId '" + resourceId + "'", e);
        }
        
        return null;
    }
    
    private Map<String, String> _getThumbnailInfo(String projectName, String resourceId)
    {
        File thumbnailFile = _onlyOfficeManager.getThumbnailFile(projectName, resourceId);
        if (thumbnailFile != null && thumbnailFile.exists())
        {
            return Map.of("thumbnail-path", URIUtils.encodePath(thumbnailFile.getAbsolutePath()));
        }
        
        return null;
    }
}
