/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum;

import java.time.ZonedDateTime;
import java.util.List;

import org.ametys.cms.data.RichText;
import org.ametys.cms.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.cms.repository.AttachableAmetysObject;
import org.ametys.cms.repository.CommentableAmetysObject;
import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.tag.TaggableAmetysObject;

/**
 * Interface representing a forum's thread
 */
public interface Thread extends ModifiableModelAwareDataAwareAmetysObject, RemovableAmetysObject, TaggableAmetysObject, AttachableAmetysObject, CommentableAmetysObject<RichTextComment>
{
    /** Attribute Threadid */
    public static final String ATTRIBUTE_ID = "id";
    /** Attribute Label */
    public static final String ATTRIBUTE_TITLE = "title";
    /** Attribute Description */
    public static final String ATTRIBUTE_CONTENT = "content";
    /** Attribute Description for editing */
    public static final String ATTRIBUTE_CONTENT_FOR_EDITING = "contentEditing";
    /** Attribute Description for editing */
    public static final String ATTRIBUTE_CONTENT_FOR_RENDERING = "contentRendering";
    /** Attribute Description abstract */
    public static final String ATTRIBUTE_CONTENT_ABSTRACT = "contentAbstract";
    /** Attribute Author */
    public static final String ATTRIBUTE_AUTHOR = "author";
    /** Attribute Creationdate */
    public static final String ATTRIBUTE_CREATIONDATE = "creationDate";
    /** Attribute Lastmodified */
    public static final String ATTRIBUTE_LASTMODIFIED = "lastModified";
    /** Attribute LastContribution : set on creation, updated for each anwer added */
    public static final String ATTRIBUTE_LASTCONTRIBUTION = "lastContribution";
    /** Attribute Category */
    public static final String ATTRIBUTE_CATEGORY = "category";
    /** Attribute Category */
    public static final String ATTRIBUTE_CLOSEAUTHOR = "closeAuthor";
    /** Attribute Category */
    public static final String ATTRIBUTE_CLOSEDATE = "closeDate";
    /** Attribute Attachments */
    public static final String ATTRIBUTE_ATTACHMENTS = "attachments";
    /** Attribute Tags */
    public static final String ATTRIBUTE_TAGS = "tags";
    /** Attribute Comments */
    public static final String ATTRIBUTE_COMMENTS = "comments";
    /** Attribute CommentsLength, used to get number of comments without parsing them */
    public static final String ATTRIBUTE_COMMENTS_LENTGH = "commentsLength";
    /** Attribute CommentAuthors, used to get number of comments without parsing them */
    public static final String ATTRIBUTE_COMMENTS_AUTHORS = "commentAuthors";
    /** Computed hasNotificationDot attribute */
    public static final String HAS_UNOPENED_THREAD_NOTIFICATION = "hasUnopenedThreadNotification";
    /** Computed acceptedAnswer attribute */
    public static final String ACCEPTED_ANSWER_COMMENT_ID = "acceptedAnswer";
    /** Computed unreadComments attribute */
    public static final String UNREAD_COMMENTS = "unreadComments";
    /** Computed isAuthor attribute */
    public static final String IS_AUTHOR = "isAuthor";
    
    /**
     * The title of the thread
     * @return The title
     */
    public String getTitle();
    
    /**
     * Set the title of the thread
     * @param title The title
     */
    public void setTitle(String title);
    
    /**
     * The content of the thread
     * @return The content
     */
    public RichText getContent();
    
    /**
     * Set the content of the thread
     * @param content The content
     */
    public void setContent(RichText content);
    
    /**
     * Get the author of the thread
     * @return the author
     */
    public UserIdentity getAuthor();
    
    /**
     * Set the author of this thread.
     * @param author the author
     */
    public void setAuthor(UserIdentity author);

    /**
     * Get the thread's creation date.
     * @return the thread's creation date.
     */
    public ZonedDateTime getCreationDate();
    
    /**
     * Set the thread's creation date.
     * @param startDate the task's creation date.
     */
    public void setCreationDate(ZonedDateTime startDate);

    /**
     * Get the thread's last modification date.
     * @return the thread's last modification date.
     */
    public ZonedDateTime getLastModified();
    
    /**
     * Set the thread's modification date.
     * @param date the last modification date to set.
     */
    public void setLastModified(ZonedDateTime date);

    /**
     * Get the thread's last contribution date.
     * @return the thread's last contribution date.
     */
    public ZonedDateTime getLastContribution();
    
    /**
     * Set the thread's contribution date.
     * @param date the last contribution date to set.
     */
    public void setLastContribution(ZonedDateTime date);

    /**
     * The category of the thread
     * @return The category
     */
    public ThreadCategoryEnum getCategory();
    
    /**
     * Set the category of the thread
     * @param category The category
     */
    public void setCategory(ThreadCategoryEnum category);
    
    /**
     * Get the close author of the thread
     * @return the close author
     */
    public UserIdentity getCloseAuthor();
    
    /**
     * Set the close author of this thread.
     * @param author the close author
     */
    public void setCloseAuthor(UserIdentity author);

    /**
     * Get the thread's close date.
     * @return the thread's close date.
     */
    public ZonedDateTime getCloseDate();
    
    /**
     * Set the thread's close date.
     * @param date the close date to set.
     */
    public void setCloseDate(ZonedDateTime date);

    /**
     * Get the accepted answer.
     * @return the accepted answer if present, null otherwise.
     */
    public RichTextComment getAcceptedAnswer();
    
    /**
     * Get the list of available comments (validated, not validated, both or none :))
     * @param includeNotValidatedComments true to get the non validated comments.
     * @param includeValidatedComments true to get the validated comment.
     * @return A non null list of comments on the ametys object.
     * @throws AmetysRepositoryException if an error occurred
     */
    public List<RichTextComment> getFlattenComments(boolean includeNotValidatedComments, boolean includeValidatedComments) throws AmetysRepositoryException;

}
