/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.indexing.project;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.indexing.explorer.AbstractSolrIndexResourceObserver;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.indexing.solr.SolrProjectResourceIndexer;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.WebConstants;

/**
 * Component for indexing {@link Project} resources.
 * Project are indexed in default and live systematically since there is currently no workflow on project resources.
 */
public class SolrIndexProjectResourceObserver extends AbstractSolrIndexResourceObserver
{
    /** The Solr indexer for project resources */
    protected SolrProjectResourceIndexer _solrProjectResourceIndexer;
    
    private Project _project;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _solrProjectResourceIndexer = (SolrProjectResourceIndexer) serviceManager.lookup(SolrProjectResourceIndexer.ROLE);
    }
    
    @Override
    protected String[] getWorkspacesToIndex()
    {
        return new String[] {RepositoryConstants.DEFAULT_WORKSPACE, WebConstants.LIVE_WORKSPACE};
    }
    
    @Override
    protected boolean isHandledResource(AmetysObject object)
    {
        AmetysObject current = object.getParent();
        while (current != null)
        {
            AmetysObject parent = current.getParent();
            if (DocumentWorkspaceModule.WORKSPACES_DOCUMENTS_NODE_NAME.equals(current.getName()) 
                    && parent != null 
                    && (RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":resources").equals(parent.getName())
                    && parent.getParent() instanceof Project)
            {
                _project = parent.getParent();
                return true;
            }
            current = current.getParent();
        }
        
        return false;
    }

    @Override
    protected void onResourceCreated(Resource resource, String workspaceName) throws Exception
    {
        if (_project != null)
        {
            // _project must be != null because #isHandledResource is called just before
            _solrProjectResourceIndexer.indexProjectResource(resource, _project, workspaceName);
            _project = null;
        }
    }

    @Override
    protected void onResourceUpdated(Resource resource, String workspaceName) throws Exception
    {
        if (_project != null)
        {
            
            // _project must be != null because #isHandledResource is called just before
            _solrProjectResourceIndexer.indexProjectResource(resource, _project, workspaceName);
            _project = null;
        }
    }

    @Override
    protected void onCollectionRenamedOrMoved(ResourceCollection resourceCollection, String workspaceName) throws Exception
    {
        if (_project != null)
        {
            // _project must be != null because #isHandledResource is called just before
            _solrProjectResourceIndexer.indexProjectResources(resourceCollection, _project, workspaceName);
            _project = null;
        }
    }
}
