/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.ProjectsCatalogueManager;
import org.ametys.plugins.workspaces.project.favorites.FavoritesHelper;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Sax the public and moderated projects of current member content
 *
 */
public class MemberProjectsGenerator extends ServiceableGenerator
{
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The project catalog manager */
    protected ProjectsCatalogueManager _projectCatalogManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** To handle favorites */
    protected FavoritesHelper _favoritesHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _projectCatalogManager = (ProjectsCatalogueManager) smanager.lookup(ProjectsCatalogueManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _favoritesHelper = (FavoritesHelper) manager.lookup(FavoritesHelper.ROLE);
    }

    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        XMLUtils.startElement(contentHandler, "projects");
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        UserIdentity user = _getUser(request);
        
        if (user != null)
        {
            // The filtered projects' categories
            Set<String> filteredCategories = _getFilteredCategories(request);
            
            // Pattern filter
            String pattern = request.getParameter("pattern");
            
            boolean excludePrivate = parameters.getParameterAsBoolean("excludePrivate", false);
            
            Map<Project, MemberType> userProjects = _projectManager.getUserProjects(user, filteredCategories, null, pattern, false, excludePrivate);
            
            Set<String> favorites = _favoritesHelper.getFavorites(user);
            
            List<Project> projects = userProjects.keySet()
                    .stream()
                    .collect(Collectors.toList());
            
            int maxMembers = parameters.getParameterAsInteger("maxMembers", Integer.MAX_VALUE);
            boolean withMembers = parameters.getParameterAsBoolean("withMembers", false);
            boolean expandGroup = parameters.getParameterAsBoolean("expandGroup", false);
            
            for (Project project : projects)
            {
                _projectCatalogManager.saxProject(contentHandler, project, withMembers, expandGroup, maxMembers, favorites.contains(project.getName()));
            }
        }
        
        XMLUtils.endElement(contentHandler, "projects");
        contentHandler.endDocument();
    }
    
    private UserIdentity _getUser(Request request)
    {
        boolean userContent = parameters.getParameterAsBoolean("userContent", false);
        if (userContent)
        {
            // Get members of user content
            Content content = (Content) request.getAttribute(Content.class.getName());
            return content.getValue("user");
        }
        else
        {
            // Get members of connected users
            return _currentUserProvider.getUser();
        }
    }
    
    private Set<String> _getFilteredCategories(Request request)
    {
        String[] categoryIds = request.getParameterValues("categories");
        if (categoryIds != null)
        {
            return Arrays.stream(categoryIds)
                    .filter(StringUtils::isNotEmpty)
                    .collect(Collectors.toSet());
        }
        else
        {
            return Set.of();
        }
    }
}
