/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members.observers;

import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.userdirectory.UserDirectoryPageHandler;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.web.cache.CacheInvalidationPolicy;
import org.ametys.web.repository.page.Page;

/**
 * {@link Observer} to invalidate cache on front-office of a user content when a member was added or removed from a project
 *
 */
public class InvalidateCacheOnMemberUpdatedObserver extends AbstractMemberObserver
{
    private CacheInvalidationPolicy _cachePolicy;
    private UserDirectoryPageHandler _udPageHandler;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _cachePolicy = (CacheInvalidationPolicy) smanager.lookup(CacheInvalidationPolicy.class.getName());
        _udPageHandler = (UserDirectoryPageHandler) smanager.lookup(UserDirectoryPageHandler.ROLE);
    }
    
    @Override
    public int getPriority()
    {
        // Will be processed after indexation
        return MAX_PRIORITY + 4000;
    }
    
    @Override
    protected void _internalObserve(Event event, List<Content> userContents) throws Exception
    {
        Set<Page> rootPages = _udPageHandler.getUserDirectoryRootPages(WorkspacesConstants.MEMBER_CONTENT_TYPE_ID);
        
        for (Content userContent : userContents)
        {
            for (Page rootPage : rootPages)
            {
                try
                {
                    _cachePolicy.invalidateCacheOnContentModification(rootPage.getSite(), userContent);
                }
                catch (Exception e)
                {
                    getLogger().error("Unable to invalidate cache for user content " + userContent, e);
                }
            }
        }
    }
}
