/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project;

import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.tika.io.FilenameUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.group.GroupDirectoryContextHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.core.util.mail.SendMailHelper.MailBuilder;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.workspaces.about.AboutWorkspaceModule;
import org.ametys.plugins.workspaces.alert.AlertWorkspaceModule;
import org.ametys.plugins.workspaces.catalog.CatalogSiteType;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryHelper;
import org.ametys.plugins.workspaces.categories.CategoryProviderExtensionPoint;
import org.ametys.plugins.workspaces.keywords.KeywordProviderExtensionPoint;
import org.ametys.plugins.workspaces.keywords.KeywordsDAO;
import org.ametys.plugins.workspaces.members.JCRProjectMember;
import org.ametys.plugins.workspaces.members.MembersWorkspaceModule;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.news.NewsWorkspaceModule;
import org.ametys.plugins.workspaces.project.favorites.FavoritesHelper;
import org.ametys.plugins.workspaces.project.notification.preferences.NotificationPreferencesHelper;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.objects.Project.InscriptionStatus;
import org.ametys.plugins.workspaces.project.rights.ProjectRightHelper;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteDAO;

import jakarta.mail.MessagingException;

/**
 * Manager for the Projects Catalogue service
 */
public class ProjectsCatalogueManager extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = ProjectsCatalogueManager.class.getName();

    /** The identifier of modules that are always active */
    public static final Set<String> DEFAULT_MODULES = Set.of(MembersWorkspaceModule.MEMBERS_MODULE_ID, AboutWorkspaceModule.ABOUT_MODULE_ID, NewsWorkspaceModule.NEWS_MODULE_ID,
            AlertWorkspaceModule.ALERT_MODULE_ID);

    /** List of allowed field received from the front */
    private static final String[] __ALLOWED_FORM_DATA = {"description", "emailList", "inscriptionStatus", "defaultProfile", "tags", "categoryTags", "keywords"};

    /** Ametys Object Resolver */
    protected AmetysObjectResolver _resolver;

    /** Current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The project members' manager */
    protected ProjectMemberManager _projectMemberManager;

    /** The right manager */
    protected RightManager _rightManager;

    /** The project manager */
    protected ProjectManager _projectManager;

    /** The source resolver */
    protected SourceResolver _sourceResolver;

    /** The site dao */
    protected SiteDAO _siteDAO;

    /** Helper for user population */
    protected PopulationContextHelper _populationContextHelper;

    /** The user manager */
    protected UserManager _userManager;

    /** Utils for i18n */
    protected I18nUtils _i18nUtils;

    /** The observation manager */
    protected ObservationManager _observationManager;

    /** Helper for group directory's context */
    protected GroupDirectoryContextHelper _groupDirectoryContextHelper;

    /** The extension point for project's categories */
    protected CategoryProviderExtensionPoint _categoryProviderEP;

    /** The extension point for project's keywords */
    protected KeywordProviderExtensionPoint _keywordProviderEP;

    /** To handle favorites */
    protected FavoritesHelper _favoritesHelper;

    /** The preference helper for notifications */
    protected NotificationPreferencesHelper _notificationPreferenceHelper;
    
    /** The user languages manager */
    protected UserLanguagesManager _userLanguagesManager;

    /** The project rights helper */
    protected ProjectRightHelper _projectRightsHelper;
    
    private CategoryHelper _categoryHelper;

    private UserHelper _userHelper;

    private KeywordsDAO _keywordsDAO;

    public void service(ServiceManager manager) throws ServiceException
    {
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _siteDAO = (SiteDAO) manager.lookup(SiteDAO.ROLE);
        _populationContextHelper = (PopulationContextHelper) manager.lookup(PopulationContextHelper.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _groupDirectoryContextHelper = (GroupDirectoryContextHelper) manager.lookup(GroupDirectoryContextHelper.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);
        _keywordProviderEP = (KeywordProviderExtensionPoint) manager.lookup(KeywordProviderExtensionPoint.ROLE);
        _categoryHelper = (CategoryHelper) manager.lookup(CategoryHelper.ROLE);
        _keywordsDAO = (KeywordsDAO) manager.lookup(KeywordsDAO.ROLE);
        _favoritesHelper = (FavoritesHelper) manager.lookup(FavoritesHelper.ROLE);
        _notificationPreferenceHelper = (NotificationPreferencesHelper) manager.lookup(NotificationPreferencesHelper.ROLE);
        _userLanguagesManager = (UserLanguagesManager) manager.lookup(UserLanguagesManager.ROLE);
        _projectRightsHelper = (ProjectRightHelper) manager.lookup(ProjectRightHelper.ROLE);
    }

    private Pair<List<String>, List<Map<String, Object>>> _createMissingKeywords(List<Object> keywords, ZoneItem zoneItem)
    {
        List<Map<String, Object>> newKeywordsInfo = Collections.emptyList();
        String[] keywordsToCreate = keywords.stream().filter(t -> t instanceof Map).map(t -> (String) ((Map) t).get("text")).toArray(String[]::new);
        if (keywordsToCreate.length > 0)
        {
            // check right
            boolean hasCatalogReadAccess = _projectRightsHelper.hasCatalogReadAccess(zoneItem);
            SitemapElement catalogPage = zoneItem.getZone().getSitemapElement();
            
            // A user need access to the catalog page, and rights either on catalog page or global context
            boolean hasRight = hasCatalogReadAccess && (_rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_HANDLE_PROJECTKEYWORDS, catalogPage) == RightResult.RIGHT_ALLOW
                    || _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_HANDLE_PROJECTKEYWORDS, "/cms") == RightResult.RIGHT_ALLOW);
            if (!hasRight)
            {
                throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " tried to create a project tag without the convinient rights");
            }

            newKeywordsInfo = _keywordsDAO.addTags(keywordsToCreate);
        }

        Iterator<Map<String, Object>> newKeywordsInfoIterator = newKeywordsInfo.iterator();
        List<String> keywordsToSet = keywords.stream().map(t -> t instanceof String ? (String) t : (String) newKeywordsInfoIterator.next().get("name")).collect(
                Collectors.toList());
        return Pair.of(keywordsToSet, newKeywordsInfo);
    }

    /**
     * Create a project
     * 
     * @param zoneItemId The id of the zoneitem holding the catalog service
     * @param title The title
     * @param description The description (can be empty)
     * @param illustration The illustration (can be a File or a local path)
     * @param category The category
     * @param keywords The project keywords
     * @param visibility The visibility
     * @param defaultProfile For public projects, profile for self registered
     *            users
     * @param language The language code
     * @param managers The managers url
     * @param modules The selected modules
     * @return Information about the new project
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> createProject(String zoneItemId, String title, String description, Object illustration, String category, List<Object> keywords, Integer visibility,
            String defaultProfile, String language, List<String> managers, List<String> modules)
    {
        ZoneItem zoneItem = _resolver.resolveById(zoneItemId);

        InscriptionStatus inscriptionStatus = visibility == 1 ? InscriptionStatus.PRIVATE : (visibility == 2 ? InscriptionStatus.MODERATED : InscriptionStatus.OPEN);
        _projectManager.checkRightsForProjectCreation(inscriptionStatus, zoneItem);

        Site catalogSite = zoneItem.getZone().getSitemapElement().getSite();
        // Get service parameters
        ModelAwareDataHolder serviceDataHolder = zoneItem.getServiceParameters();
        String titlePrefix = serviceDataHolder.getValue("titlePrefix", false, "");
        String urlPrefix = catalogSite.getValue(CatalogSiteType.PROJECT_URL_PREFIX_SITE_PARAM, false, "");
        String[] availableLanguages = serviceDataHolder.getValue("availableLanguages");
        String skin = catalogSite.getValue(CatalogSiteType.PROJECT_DEFAULT_SKIN_SITE_PARAM, false, "");
        String forceAcceptCookie = serviceDataHolder.getValue("force_accept_cookies", false, "");
        String[] populationIds = serviceDataHolder.getValue("populationIds") != null ? serviceDataHolder.getValue("populationIds") : new String[0];


        // Check language
        if (Arrays.stream(availableLanguages).filter(language::equals).findFirst().isEmpty())
        {
            throw new IllegalArgumentException("Cannot create project with language '" + language + "' since it is not part of the available languages " + availableLanguages);
        }
        // Check profile
        String defaultManagerProfile = Config.getInstance().getValue("workspaces.profile.managerdefault", false, null);
        if (StringUtils.isBlank(defaultManagerProfile))
        {
            throw new IllegalArgumentException("The general configuration parameter 'workspaces.profile.managerdefault' cannot be empty");
        }

        // Create tags if necessary
        Pair<List<String>, List<Map<String, Object>>> keywordsInfo = _createMissingKeywords(keywords, zoneItem);
        List<String> keywordsToSet = keywordsInfo.getLeft();
        List<Map<String, Object>> newKeywordsInfo = keywordsInfo.getRight();

        // Create project object in repo
        Map<String, Object> additionalValues = new HashMap<>();
        additionalValues.put("description", StringUtils.defaultString(description));
        additionalValues.put("categoryTags", Collections.singletonList(category));
        additionalValues.put("inscriptionStatus", inscriptionStatus.toString());
        additionalValues.put("defaultProfile", defaultProfile);
        additionalValues.put("keywords", keywordsToSet);
        additionalValues.put("language", language);

        List<String> errors = new ArrayList<>();
        String prefixedTitle = (titlePrefix + " " + title).trim();
        Project project = _projectManager.createProject(_findName(prefixedTitle), prefixedTitle, additionalValues, _withDefaultModules(modules), errors);

        if (!CollectionUtils.isEmpty(errors))
        {
            Map<String, Object> result = new HashMap<>();
            result.put("success", false);

            return result;
        }

        // Add site infos
        _updateSiteInfos(project, urlPrefix, skin, forceAcceptCookie, catalogSite, illustration, language);

        // Assign populations and manager
        _assignPopulations(project, catalogSite, Set.of(populationIds));
        _assignManagers(project, managers, defaultManagerProfile);
        project.saveChanges();

        Map<String, Object> result = new HashMap<>();
        result.put("success", true);
        result.put("project", _detailedMyProject2json(project, zoneItem.getZone().getSitemapElement(), false,
                _notificationPreferenceHelper.getPausedProjects(_currentUserProvider.getUser()) == null ? null : false));
        result.put("keywords", newKeywordsInfo);
        return result;
    }

    private void _assignPopulations(Project project, Site catalogSite, Set<String> serviceParameterPopulationIds)
    {
        Site site = project.getSite();

        // By default, copy populations from the catalog
        Set<String> populations = _populationContextHelper.getUserPopulationsOnContext("/sites/" + catalogSite.getName(), false);
        Set<String> frontPopulations = _populationContextHelper.getUserPopulationsOnContext("/sites-fo/" + catalogSite.getName(), false);

        // If populations have been specified in the service, use them
        if (serviceParameterPopulationIds.size() > 0)
        {
            populations = populations.stream().filter(populationId -> serviceParameterPopulationIds.contains(populationId)).collect(Collectors.toSet());
            frontPopulations = frontPopulations.stream().filter(populationId -> serviceParameterPopulationIds.contains(populationId)).collect(Collectors.toSet());
        }

        _populationContextHelper.link("/sites/" + site.getName(), populations);
        _populationContextHelper.link("/sites-fo/" + site.getName(), frontPopulations);

        Set<String> groupDirectories = _groupDirectoryContextHelper.getGroupDirectoriesOnContext("/sites/" + catalogSite.getName());
        Set<String> frontGroupDirectories = _groupDirectoryContextHelper.getGroupDirectoriesOnContext("/sites-fo/" + catalogSite.getName());

        _groupDirectoryContextHelper.link("/sites/" + site.getName(), new ArrayList<>(groupDirectories));
        _groupDirectoryContextHelper.link("/sites-fo/" + site.getName(), new ArrayList<>(frontGroupDirectories));
    }

    private void _assignManagers(Project project, List<String> managers, String defaultManagerProfile)
    {
        List<UserIdentity> projectManagers = managers.stream().map(UserIdentity::stringToUserIdentity).filter(Objects::nonNull).filter(
                user -> _projectManager.isUserInProjectPopulations(project, user)).collect(Collectors.toList());

        _projectMemberManager.setProjectManager(project.getName(), defaultManagerProfile, projectManagers);
    }

    private void _updateSiteInfos(Project project, String urlPrefix, String skin, String forceAcceptCookie, Site catalogSite, Object illustration, String language)
    {
        Site site = project.getSite();

        site.setUrl(urlPrefix + "/" + project.getName());
        site.setValue("skin", skin);
        if (site.hasDefinition("force-accept-cookies"))
        {
            site.setValue("force-accept-cookies", forceAcceptCookie);
        }

        site.setValue("display-restricted-pages", false);
        site.setValue("ping_activated", false);
        site.setValue("site-mail-from", catalogSite.getValue("site-mail-from"));
        site.setValue("site-contents-comments-postvalidation", catalogSite.getValue("site-contents-comments-postvalidation"));
        site.setValue("color", catalogSite.getValue("color"));

        _setIllustration(site, illustration);

        _siteDAO.setLanguages(site, Collections.singletonList(language));

        site.saveChanges();

        Map<String, Object> eventParams = new HashMap<>();
        eventParams = new HashMap<>();
        eventParams.put(org.ametys.web.ObservationConstants.ARGS_SITE, site);
        _observationManager.notify(new Event(org.ametys.web.ObservationConstants.EVENT_SITE_UPDATED, _currentUserProvider.getUser(), eventParams));
    }

    private void _setIllustration(Site site, Object illustration)
    {
        Object illustrationObject = null;
        try
        {
            illustrationObject = _getIllustrationSource(illustration);
            if (illustrationObject instanceof Source)
            {
                Source illustrationSource = (Source) illustrationObject;
                try (InputStream is = illustrationSource.getInputStream())
                {
                    site.setIllustration(is, illustrationSource.getMimeType(), FilenameUtils.getName(illustrationSource.getURI()), ZonedDateTime.now());
                }
            }
            else if (illustrationObject instanceof Part)
            {
                Part illustrationPart = (Part) illustrationObject;
                try (InputStream is = illustrationPart.getInputStream())
                {
                    site.setIllustration(is, illustrationPart.getMimeType(), illustrationPart.getUploadName(), ZonedDateTime.now());
                }
            }
        }
        catch (IOException e)
        {
            throw new IllegalArgumentException("Cannot not get illustration", e);
        }
        finally
        {
            if (illustrationObject instanceof Source)
            {
                _sourceResolver.release((Source) illustrationObject);
            }
        }

    }

    private Object _getIllustrationSource(Object illustration) throws IOException
    {
        if (illustration instanceof String)
        {
            String illustrationAsString = (String) illustration;
            if (illustrationAsString.contains("/") || illustrationAsString.contains("\\"))
            {
                throw new IllegalArgumentException("Cannot choose an illustration outside the library directory");
            }

            return _sourceResolver.resolveURI("plugin:workspaces://resources/img/catalog/library/" + illustrationAsString);
        }
        else if (illustration instanceof Part)
        {
            Part illustrationAsFile = (Part) illustration;
            return illustrationAsFile;
        }
        else // boolean => unchanged
        {
            return null;
        }
    }

    private Set<String> _withDefaultModules(List<String> modules)
    {
        Set<String> modulesToActivate = new HashSet<>();

        modulesToActivate.addAll(DEFAULT_MODULES);
        modulesToActivate.addAll(modules);

        return modulesToActivate;
    }

    private String _findName(String title)
    {
        String originalName = NameHelper.filterName(title);
        String name = originalName;

        int index = 2;
        while (_projectManager.hasProject(name))
        {
            name = originalName + "-" + (index++);
        }

        return name;
    }

    /**
     * Edit an existing project
     * 
     * @param zoneItemId The id of the zoneitem holding the catalog service
     * @param projectId The id of the project
     * @param title New title
     * @param description New description
     * @param illustration New illustration
     * @param category New category
     * @param keywords The project keywords
     * @param visibility New visibility
     * @param defaultProfile New default profile
     * @param managers New managers
     * @param modules New modules
     * @return The success map with project description
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> editProject(String zoneItemId, String projectId, String title, String description, Object illustration, String category, List<Object> keywords,
            Integer visibility, String defaultProfile, List<String> managers, List<String> modules)
    {
        Project project = _resolver.resolveById(projectId);
        if (project == null)
        {
            throw new IllegalArgumentException("Unable to edit a project, invalid project id received '" + projectId + "'");
        }

        ZoneItem zoneItem = _resolver.resolveById(zoneItemId);
        InscriptionStatus inscriptionStatus = visibility == 1 ? InscriptionStatus.PRIVATE : (visibility == 2 ? InscriptionStatus.MODERATED : InscriptionStatus.OPEN);
        _projectManager.checkRightsForProjectEdition(project, inscriptionStatus, zoneItem);

        // Check profile
        String defaultManagerProfile = Config.getInstance().getValue("workspaces.profile.managerdefault", false, null);
        if (StringUtils.isBlank(defaultManagerProfile))
        {
            throw new IllegalArgumentException("The general configuration parameter 'workspaces.profile.managerdefault' cannot be empty");
        }

        boolean canEdit = _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_EDIT, project) == RightResult.RIGHT_ALLOW
                || _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_EDIT, zoneItem.getZone().getSitemapElement()) == RightResult.RIGHT_ALLOW;
        if (!canEdit)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to edit the project '" + projectId + "' without sufficient rights");
        }

        // Create keywords if necessary
        Pair<List<String>, List<Map<String, Object>>> keywordsInfo = _createMissingKeywords(keywords, zoneItem);
        List<String> keywordsToSet = keywordsInfo.getLeft();
        List<Map<String, Object>> newKeywordsInfo = keywordsInfo.getRight();

        project.setTitle(title);
        project.setDescription(StringUtils.defaultString(description));
        project.setInscriptionStatus(inscriptionStatus.toString());
        project.setDefaultProfile(defaultProfile);
        project.setCategoryTags(Collections.singletonList(category));
        project.setKeywords(keywordsToSet.toArray(new String[keywordsToSet.size()]));

        Site site = project.getSite();

        _projectManager.setProjectSiteTitle(site, project.getTitle());
        _setIllustration(site, illustration);
        
        if (site.needsSave())
        {
            site.saveChanges();

            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_SITE, site);
            _observationManager.notify(new Event(ObservationConstants.EVENT_SITE_UPDATED, _currentUserProvider.getUser(), eventParams));
        }

        _updateModules(project, _withDefaultModules(modules));
        _assignManagers(project, managers, defaultManagerProfile);

        if (project.needsSave())
        {
            project.saveChanges();
        }
        
        // Notify observers
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.plugins.workspaces.ObservationConstants.ARGS_PROJECT, project);
        eventParams.put(org.ametys.plugins.workspaces.ObservationConstants.ARGS_PROJECT_ID, project.getId());
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.ObservationConstants.EVENT_PROJECT_UPDATED, _currentUserProvider.getUser(), eventParams));

        UserIdentity user = _currentUserProvider.getUser();

        Map<String, Object> result = new HashMap<>();
        result.put("success", true);
        result.put("project", _detailedMyProject2json(project, zoneItem.getZone().getSitemapElement(), _favoritesHelper.getFavorites(user).contains(project.getName()),
                _notificationPreferenceHelper.getPausedProjects(user).contains(project.getName())));
        result.put("keywords", newKeywordsInfo);
        return result;
    }

    private void _updateModules(Project project, Set<String> modules)
    {
        Set<String> modulesToActivate = new HashSet<>(modules);
        Set<String> modulesToDeactivate = new HashSet<>(Arrays.asList(project.getModules()));
        modulesToActivate.removeAll(new HashSet<>(Arrays.asList(project.getModules())));
        modulesToDeactivate.removeAll(modules);

        if (!modulesToActivate.isEmpty())
        {
            Map<String, Object> additionalValues = new HashMap<>();
            additionalValues.put("description", project.getDescription());
            additionalValues.put("inscriptionStatus", project.getInscriptionStatus());
            additionalValues.put("emailList", project.getMailingList());
            additionalValues.put("defaultProfile", project.getDefaultProfile());
            additionalValues.put("categoryTags", project.getCategories());
            additionalValues.put("keywords", project.getKeywords());

            Site site = project.getSite();
            if (site != null)
            {
                additionalValues.put("language", site.getSitemaps().iterator().next().getName());
            }

            _projectManager.activateModules(project, modulesToActivate, additionalValues);
        }
        if (!modulesToDeactivate.isEmpty())
        {
            _projectManager.deactivateModules(project, modulesToDeactivate);
        }
    }

    /**
     * Delete a project
     * 
     * @param zoneItemId The id of the zoneitem holding the catalog service
     * @param projectId The project id
     * @return The result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteProject(String zoneItemId, String projectId)
    {
        Project project = _resolver.resolveById(projectId);

        if (project == null)
        {
            throw new IllegalArgumentException("Unable to delete a project, invalid project id received '" + projectId + "'");
        }

        ZoneItem zoneItem = _resolver.resolveById(zoneItemId);
        
        boolean canDelete = _projectRightsHelper.hasCatalogReadAccess(zoneItem)
                && (_rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_DELETE, project) == RightResult.RIGHT_ALLOW
                || _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_DELETE, zoneItem.getZone().getSitemapElement()) == RightResult.RIGHT_ALLOW);
        if (!canDelete)
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to delete the project '" + projectId + "' without sufficient rights");
        }

        Map<String, Object> result = new HashMap<>();
        result.put("sites", _projectManager.deleteProject(project));
        result.put("success", true);
        return result;
    }

    /**
     * Add the current user to the project, if the project's inscriptions are
     * opened
     * 
     * @param projectId The project id
     * @return The result
     * @throws MessagingException If an error occurred sending a notification
     *             mail to the project manager
     * @throws IOException If an error occurred sending the email to the
     *             project's manager
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> joinProject(String projectId) throws MessagingException, IOException
    {
        Map<String, Object> result = new HashMap<>();
        Project project = _resolver.resolveById(projectId);

        if (project == null)
        {
            throw new IllegalArgumentException("Unable to join a project, invalid project id received '" + projectId + "'");
        }

        UserIdentity currentUser = _currentUserProvider.getUser();

        Site site = project.getSite();
        try
        {
            if (!_projectManager.isUserInProjectPopulations(project, currentUser))
            {
                // User is not in the project site populations, cannot be added
                result.put("success", false);
                return result;
            }
        }
        catch (IllegalArgumentException e)
        {
            result.put("success", false);
            return result;
        }

        JCRProjectMember addedMember = _projectMemberManager.addProjectMember(project, currentUser);
        boolean success = addedMember != null;

        result.put("success", success);
        if (success)
        {
            String url = site.getUrl();
            result.put("url", url);

            String mailFrom = Config.getInstance().getValue("smtp.mail.from");
            
            if (mailFrom != null)
            {
                Map<String, I18nizableTextParameter> params = new HashMap<>();
                User current = _userManager.getUser(currentUser);
                params.put("user", new I18nizableText(current != null ? current.getFullName() : currentUser.getLogin()));
                params.put("project", new I18nizableText(project.getTitle()));
                params.put("url", new I18nizableText(url != null ? url : ""));
                
                I18nizableText i18nSubject = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOGUE_JOINPROJECT_MAIL_TITLE", params);
                I18nizableText i18nTextBody = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOGUE_JOINPROJECT_MAIL_BODY_TEXT", params);
                
                MailBodyBuilder htmlBodyBuilder = StandardMailBodyHelper.newHTMLBody()
                        .withTitle(i18nSubject)
                        .withMessage(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOGUE_JOINPROJECT_MAIL_BODY_HTML", params))
                        .withLink(url, new I18nizableText("plugin.workspaces", "PROJECT_MAIL_NOTIFICATION_BODY_DEFAULT_BUTTON_TEXT"));
                
                _sendMailToManagers(project, mailFrom, i18nSubject, htmlBodyBuilder, i18nTextBody);
            }
        }
        return result;
    }

    /**
     * Send a demand to join a project to the project's manager, if the
     * project's inscriptions are moderated
     * 
     * @param projectId The project to join
     * @param message A message to send to the project's manager.
     * @return The result
     * @throws MessagingException If an error occurred sending the email to the
     *             project's manager
     * @throws IOException If an error occurred sending the email to the
     *             project's manager
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> askToJoinProject(String projectId, String message) throws MessagingException, IOException
    {
        Map<String, Object> result = new HashMap<>();

        Project project = _resolver.resolveById(projectId);
        if (project == null)
        {
            throw new IllegalArgumentException("Unable to join a project, invalid project id received '" + projectId + "'");
        }

        UserIdentity currentUser = _currentUserProvider.getUser();

        if (!_projectManager.isUserInProjectPopulations(project, currentUser))
        {
            // User is not in the project site populations, cannot be added
            result.put("success", false);
            return result;
        }

        _sendAskToJoinMail(message, project, currentUser);
        result.put("success", true);
        result.put("added-notification", Config.getInstance().getValue("workspaces.member.added.send.notification"));
        return result;
    }

    private void _sendAskToJoinMail(String message, Project project, UserIdentity joiningUser) throws MessagingException, IOException
    {
        String url = getAddUserUrl(project, joiningUser);

        String mailFrom = Config.getInstance().getValue("smtp.mail.from");
        
        if (mailFrom != null)
        {
            Map<String, I18nizableTextParameter> params = new HashMap<>();
            User current = _userManager.getUser(joiningUser);
            params.put("user", new I18nizableText(current != null ? current.getFullName() : joiningUser.getLogin()));
            params.put("project", new I18nizableText(project.getTitle()));
            params.put("url", new I18nizableText(url));
            
            I18nizableText i18nSubject = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOGUE_JOINPROJECT_ASK_MAIL_SUBJECT", params);
            
            MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withTitle(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOGUE_JOINPROJECT_ASK_MAIL_BODY_TITLE", params))
                .withMessage(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOGUE_JOINPROJECT_ASK_MAIL_BODY", params))
                .withLink(url, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOGUE_JOINPROJECT_ASK_MAIL_ADD_USER_LINK"));
            
            if (!StringUtils.isEmpty(message))
            {
                bodyBuilder.withDetails(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOGUE_JOINPROJECT_ASK_MAIL_BODY_MSG"), message, false);
            }
            
            _sendMailToManagers(project, mailFrom, i18nSubject, bodyBuilder, null);
        }
    }
    
    private void _sendMailToManagers(Project project, String mailFrom, I18nizableText i18nSubject, MailBodyBuilder bodyBuilder, I18nizableText i18nTextBody) throws MessagingException, IOException
    {
        String defaultLanguage = _userLanguagesManager.getDefaultLanguage();
        Map<String, List<String>> managersToNotifyByLanguage = Arrays.stream(project.getManagers())
                .map(manager -> _userManager.getUser(manager))
                .filter(Objects::nonNull)
                .map(user -> Pair.of(user, user.getEmail()))
                .filter(p -> StringUtils.isNotEmpty(p.getRight()))
                .collect(Collectors.groupingBy(
                        p -> {
                            return StringUtils.defaultIfBlank(p.getLeft().getLanguage(), defaultLanguage);
                        },
                        Collectors.mapping(
                                Pair::getRight,
                                Collectors.toList()
                        )
                    )
                );

        for (String lang : managersToNotifyByLanguage.keySet())
        {
            // If no user language, user project site one
            String subject = _i18nUtils.translate(i18nSubject, lang);
            
            String htmlBody = bodyBuilder.withLanguage(lang).build();
            
            MailBuilder mail = SendMailHelper.newMail()
                .withSubject(subject)
                .withHTMLBody(htmlBody)
                .withSender(mailFrom)
                .withRecipients(managersToNotifyByLanguage.get(lang));
                
            if (i18nTextBody != null)
            {
                String textBody = _i18nUtils.translate(i18nTextBody, lang);
                mail.withTextBody(textBody);
            }
            
            mail.sendMail();
        }
    }

    /**
     * Get the absolute url to add a user to a project
     * 
     * @param project The project
     * @param user the identity of user to add
     * @return the absolute page url
     */
    protected String getAddUserUrl(Project project, UserIdentity user)
    {
        String memberPage = "";

        Set<Page> membersPages = _projectManager.getModulePages(project, MembersWorkspaceModule.MEMBERS_MODULE_ID);
        if (!membersPages.isEmpty())
        {
            memberPage = ResolveURIComponent.resolve("page", membersPages.iterator().next().getId(), false, true);
        }

        Site site = project.getSite();
        String siteURL = site.getUrl();
        String urlWithoutScheme = StringUtils.substringAfter(siteURL, "://");
        String relativeURL = StringUtils.contains(urlWithoutScheme, "/") ? "/" + StringUtils.substringAfter(urlWithoutScheme, "/") : "";
        return siteURL + "/_authenticate?requestedURL="
                + URIUtils.encodeParameter(relativeURL + "/plugins/workspaces/add-member?redirect="
                        + URIUtils.encodeParameter(memberPage + "?added=" + URIUtils.encodeParameter(UserIdentity.userIdentityToString(user))) + "&user="
                        + URIUtils.encodeParameter(UserIdentity.userIdentityToString(user)) + "&project=" + URIUtils.encodeParameter(project.getName()));
    }

    /**
     * Get the list of allowed data in the form
     * 
     * @return the list of allowed data in the form
     */
    protected String[] getAllowedFormData()
    {
        return __ALLOWED_FORM_DATA;
    }

    /**
     * Callable to get projects of the user and the public projects he can
     * subscribe.
     * 
     * @param zoneItemId the zoneItemId of the catalog service, used to get
     *            allowed populations
     * @return A map with three entries an entry for user projects, another one
     *         for public projects and finally one for the project's creation
     *         right
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getUserAndPublicProjects(String zoneItemId)
    {
        Map<String, Object> result = new HashMap<>();

        ZoneItem zoneItem = _resolver.resolveById(zoneItemId);
        if (!_projectRightsHelper.hasCatalogReadAccess(zoneItem))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to do read operation without convenient right");
        }

        ModelAwareDataHolder serviceDataHolder = zoneItem.getServiceParameters();
        String[] populationIds = serviceDataHolder.getValue("populationIds") != null ? serviceDataHolder.getValue("populationIds") : new String[0];

        UserIdentity user = _currentUserProvider.getUser();

        // Check if the user is inside populations of future workspaces (only if
        // at least one population have been selected)
        boolean inPopulation = populationIds.length == 0 || Arrays.asList(populationIds).contains(user.getPopulationId());

        SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
        boolean canCreatePrivateProjet = inPopulation && (_rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PRIVATE, "/cms") == RightResult.RIGHT_ALLOW
                || _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PRIVATE, sitemapElement) == RightResult.RIGHT_ALLOW);
        boolean canCreatePublicProjetWithModeration = inPopulation
                && (_rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_MODERATED, "/cms") == RightResult.RIGHT_ALLOW
                        || _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_MODERATED, sitemapElement) == RightResult.RIGHT_ALLOW);
        boolean canCreatePublicProjet = inPopulation && (_rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_OPENED, "/cms") == RightResult.RIGHT_ALLOW
                || _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_CREATE_PUBLIC_OPENED, sitemapElement) == RightResult.RIGHT_ALLOW);

        result.put("canCreate", canCreatePrivateProjet || canCreatePublicProjetWithModeration || canCreatePublicProjet);
        result.put("canCreatePrivateProject", canCreatePrivateProjet);
        result.put("canCreatePublicProjectWithModeration", canCreatePublicProjetWithModeration);
        result.put("canCreatePublicProject", canCreatePublicProjet);

        List<Map<String, Object>> userProjects = new ArrayList<>();
        List<Map<String, Object>> publicProjects = new ArrayList<>();

        Set<String> favorites = _favoritesHelper.getFavorites(user);
        Set<String> pausedProjects = _notificationPreferenceHelper.getPausedProjects(user);
        for (Project project : _projectManager.getProjects())
        {
            if (_projectMemberManager.isProjectMember(project, user))
            {
                Map<String, Object> json = _detailedMyProject2json(project, sitemapElement, favorites.contains(project.getName()),
                        pausedProjects != null ? pausedProjects.contains(project.getName()) : null);
                userProjects.add(json);
            }
            else if (project.getInscriptionStatus() != InscriptionStatus.PRIVATE && _projectManager.isUserInProjectPopulations(project, user))
            {
                Map<String, Object> json = detailedProject2json(project);
                publicProjects.add(json);
            }
        }

        result.put("userProjects", userProjects);
        result.put("availablePublicProjects", publicProjects);

        return result;
    }


    /**
     * Get a project by its name
     * @param projectName the project name
     * @param zoneItemId The id of the zoneitem holding the catalog service
     * @return a json map representing the project
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getProjectByName(String projectName, String zoneItemId)
    {
        ZoneItem zoneItem = _resolver.resolveById(zoneItemId);
        if (!_projectRightsHelper.hasCatalogReadAccess(zoneItem))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to do read operation without convenient right");
        }

        Project project = _projectManager.getProject(projectName);

        return detailedProject2json(project);
    }
    /**
     * Callable to get projects of the user.
     * 
     * @return A map with the user projects
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> getUserProjects()
    {
        UserIdentity user = _currentUserProvider.getUser();

        List<Map<String, Object>> userProjects = new ArrayList<>();

        Set<String> favorites = _favoritesHelper.getFavorites(user);
        Set<String> pausedProjects = _notificationPreferenceHelper.getPausedProjects(user);
        for (Project project : _projectManager.getProjects())
        {
            if (_projectMemberManager.isProjectMember(project, user))
            {
                // Put null for the catalog page because this method is only
                // used for mobile app and mobile app ignore canEdit and
                // canDelete rights
                Map<String, Object> json = _detailedMyProject2json(project, null, favorites.contains(project.getName()),
                        pausedProjects != null ? pausedProjects.contains(project.getName()) : null);
                userProjects.add(json);
            }
        }

        return userProjects;
    }

    private Map<String, Object> _detailedMyProject2json(Project project, SitemapElement catalogPage, boolean isFavorite, Boolean isPaused)
    {
        Map<String, Object> json = detailedProject2json(project);
        json.put("favorite", isFavorite);
        json.put("notification", isPaused != null ? !isPaused : null);
        boolean canEdit = _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_EDIT, project) == RightResult.RIGHT_ALLOW
                || catalogPage != null && _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_EDIT, catalogPage) == RightResult.RIGHT_ALLOW;
        json.put("canEdit", canEdit);
        boolean canDelete = _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_DELETE, project) == RightResult.RIGHT_ALLOW
                || catalogPage != null && _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_DELETE, catalogPage) == RightResult.RIGHT_ALLOW;
        json.put("canDelete", canDelete);
        json.put("canAccessBO", _projectManager.canAccessBO(project));
        json.put("canLeaveProject", _projectManager.canLeaveProject(project));
        return json;
    }

    /**
     * Transform a {@link Project} into a json map
     * 
     * @param project the project to parse
     * @return a json map
     */
    public Map<String, Object> detailedProject2json(Project project)
    {
        Map<String, Object> json = new HashMap<>();

        json.put("id", project.getId());
        json.put("name", project.getName());
        json.put("title", project.getTitle());
        json.put("url", _projectManager.getProjectUrl(project, StringUtils.EMPTY));

        json.put("defaultProfile", project.getDefaultProfile());

        Set<String> categories = project.getCategories();
        if (categories.size() != 1)
        {
            getLogger().warn("Project " + project.getTitle() + " (" + project.getId() + ") should have one and only one category");
        }

        if (!categories.isEmpty())
        {
            String c = categories.iterator().next();
            Category category = _categoryProviderEP.getTag(c, new HashMap<>());

            if (category != null)
            {
                Map<String, Object> map = new HashMap<>();
                map.put("id", category.getId());
                map.put("name", category.getName());
                map.put("title", category.getTitle());
                map.put("color", _categoryHelper.getCategoryColor(category).get("main"));

                json.put("category", map);
            }
        }

        String[] keywords = Stream.of(project.getKeywords()).filter(k -> _keywordProviderEP.hasTag(k, new HashMap<>())).toArray(String[]::new);
        json.put("keywords", keywords);

        UserIdentity[] managers = project.getManagers();
        if (managers.length > 0)
        {
            json.put("managers", Arrays.stream(managers).map(_userHelper::user2json).filter(userAsJson -> !userAsJson.equals(Collections.EMPTY_MAP)).collect(Collectors.toList()));
        }

        List<Map<String, Object>> members = new ArrayList<>();
        for (int i = 1; i < Math.min(managers.length, 4); i++)
        {
            members.add(_userHelper.user2json(managers[i]));
        }

        if (members.size() < 3)
        {
            List<UserIdentity> managersList = Arrays.asList(managers);

            members.addAll(_projectMemberManager.getProjectMembers(project, true).stream().map(ProjectMember::getUser).map(User::getIdentity).filter(
                    Predicate.not(managersList::contains)).limit(3 - members.size()).map(_userHelper::user2json).collect(Collectors.toList()));
        }

        json.put("members", members);
        json.put("membersCount", _projectMemberManager.getMembersCount(project));

        json.put("modules", Arrays.asList(project.getModules()));

        switch (project.getInscriptionStatus())
        {
            case PRIVATE:
                json.put("visibility", 1);
                break;
            case MODERATED:
                json.put("visibility", 2);
                break;
            case OPEN:
                // fallthrought
            default:
                json.put("visibility", 3);
                break;
        }

        json.put("description", project.getDescription());

        Site site = project.getSite();
        if (site != null)
        {
            json.put("site", site.getName());
            json.put("language", site.getSitemaps().iterator().next().getName());

            if (site.getIllustration() != null)
            {
                String illustration = ResolveURIComponent.resolveCroppedImage("site-parameter", site.getName() + ";illustration", 252, 389, false, true);
                json.put("illustration", illustration);
            }
        }

        return json;
    }

    /**
     * SAX a project
     * 
     * @param contentHandler The content handler to sax into
     * @param project the project
     * @throws SAXException if an error occurred while saxing
     */
    public void saxProject(ContentHandler contentHandler, Project project) throws SAXException
    {
        saxProject(contentHandler, project, false, false, 0, false);
    }

    /**
     * SAX a project
     * 
     * @param contentHandler The content handler to sax into
     * @param project the project
     * @param withMembers true to sax members
     * @param expandGroup true to expand group members
     * @param maxMembers the max number of members to sax. Set to -1 to sax all
     *            members
     * @param isFavorite true if the project is in user favorites
     * @throws SAXException if an error occurred while saxing
     */
    public void saxProject(ContentHandler contentHandler, Project project, boolean withMembers, boolean expandGroup, int maxMembers, boolean isFavorite) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", project.getId());
        attrs.addCDATAAttribute("name", project.getName());
        attrs.addCDATAAttribute("favorite", String.valueOf(isFavorite));

        XMLUtils.startElement(contentHandler, "project", attrs);

        XMLUtils.createElement(contentHandler, "title", project.getTitle());

        XMLUtils.createElement(contentHandler, "inscriptionStatus", project.getInscriptionStatus().name());

        String description = project.getDescription();
        if (description != null)
        {
            XMLUtils.createElement(contentHandler, "description", description);
        }
        XMLUtils.createElement(contentHandler, "url", _projectManager.getProjectUrl(project, StringUtils.EMPTY));

        saxCategory(contentHandler, project);

        for (String keyword : project.getKeywords())
        {
            XMLUtils.createElement(contentHandler, "keyword", keyword);
        }

        UserIdentity[] managers = project.getManagers();
        if (managers.length > 0)
        {
            for (UserIdentity userIdentity : managers)
            {
                _userHelper.saxUserIdentity(userIdentity, contentHandler, "manager");
            }
        }

        if (withMembers)
        {
            Set<ProjectMember> members = _projectMemberManager.getProjectMembers(project, expandGroup);

            int nbMembers = members.size();

            attrs.clear();
            attrs.addCDATAAttribute("total", String.valueOf(nbMembers));
            XMLUtils.startElement(contentHandler, "members", attrs);

            int count = 0;
            Iterator<ProjectMember> it = members.iterator();
            while (it.hasNext() && (maxMembers < 0 || count < maxMembers))
            {
                ProjectMember member = it.next();
                if (!member.isManager())
                {
                    _userHelper.saxUser(member.getUser(), contentHandler, "member");
                    count++;
                }
            }

            XMLUtils.endElement(contentHandler, "members");
        }

        Site site = project.getSite();
        if (site != null)
        {
            attrs.clear();
            attrs.addCDATAAttribute("name", site.getName());
            attrs.addCDATAAttribute("language", site.getSitemaps().iterator().next().getName());

            XMLUtils.createElement(contentHandler, "site", attrs);
        }

        XMLUtils.endElement(contentHandler, "project");
    }

    /**
     * SAX the project's category
     * 
     * @param contentHandler the content handler to sax into
     * @param project the project
     * @throws SAXException if an error occurred while saxing
     */
    public void saxCategory(ContentHandler contentHandler, Project project) throws SAXException
    {
        saxCategory(contentHandler, project, "category");
    }

    /**
     * SAX the project's category
     * 
     * @param contentHandler the content handler to sax into
     * @param project the project
     * @param tagName the tag name for category
     * @throws SAXException if an error occurred while saxing
     */
    public void saxCategory(ContentHandler contentHandler, Project project, String tagName) throws SAXException
    {
        Set<String> categories = project.getCategories();
        if (!categories.isEmpty())
        {
            String c = categories.iterator().next();
            Category category = _categoryProviderEP.getTag(c, new HashMap<>());

            if (category != null)
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", category.getId());
                attrs.addCDATAAttribute("name", category.getName());
                attrs.addCDATAAttribute("color", _categoryHelper.getCategoryColor(category).get("main"));

                XMLUtils.startElement(contentHandler, tagName, attrs);
                category.getTitle().toSAX(contentHandler);
                XMLUtils.endElement(contentHandler, tagName);
            }
        }
    }
}
