/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.enumerators;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.web.WebHelper;

/**
 * Enumerator for the populations available to a project
 */
public class ProjectPopulationEnumerator extends AbstractLogEnabled implements Enumerator<String>, Serviceable, Contextualizable
{
    /** The user population DAO */
    private UserPopulationDAO _userPopulationDAO;
    /** The helper for the associations population/context */
    private PopulationContextHelper _populationContextHelper;
    /** The avalon context */
    private Context _context;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _userPopulationDAO = (UserPopulationDAO) serviceManager.lookup(UserPopulationDAO.ROLE);
        _populationContextHelper = (PopulationContextHelper) serviceManager.lookup(PopulationContextHelper.ROLE);
    }

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public I18nizableText getEntry(String code) throws Exception
    {
        UserPopulation population = _userPopulationDAO.getUserPopulation(code);
        return population != null ? population.getLabel() : null;
    }

    public Map<String, I18nizableText> getEntries() throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        String siteName = WebHelper.getSiteName(request);
        
        Set<String> populationIds = _populationContextHelper.getUserPopulationsOnContexts(List.of("/sites/" + siteName, "/sites-fo/" + siteName), false);
        List<UserPopulation> populations = populationIds.stream().map(pId -> _userPopulationDAO.getUserPopulation(pId)).collect(Collectors.toList());

        return populations.stream().collect(Collectors.toMap(UserPopulation::getId, UserPopulation::getLabel));
    }
    
}
