/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.generators;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.components.source.SourceUtil;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceNotFoundException;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.plugins.core.ui.glyph.CssFontHelper;
import org.ametys.plugins.messagingconnector.EventRecurrenceTypeEnum;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.CalendarWorkspaceModule;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.PluginsManager;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Generator for the vuejs calendar service
 */
public class ProjectsCalendarGenerator extends ServiceableGenerator implements Contextualizable
{
    private SourceResolver _sourceResolver;

    /** The avalon context */
    private Context _context;
    
    private SiteManager _siteManager;
    private ProjectManager _projectManager;
    /** The workspace module EP */
    private WorkspaceModuleExtensionPoint _workspaceModuleEP;
    private CssFontHelper _cssFontHelper;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _sourceResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _workspaceModuleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _cssFontHelper = (CssFontHelper) manager.lookup(CssFontHelper.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "calendar");

        _saxColors();
        _saxAllColors();
        _saxResourceCalendar();
        _saxRecurrenceTypes();
        _saxGlyphs();
        
        XMLUtils.endElement(contentHandler, "calendar");
        contentHandler.endDocument();
    }

    private void _saxColors() throws SAXException, MalformedURLException, IOException, ProcessingException
    {
        Source calendarColors = null;
        try
        {
            calendarColors = _sourceResolver.resolveURI("plugin:workspaces://calendar-colors.xml");
            if (calendarColors.exists())
            {
                SourceUtil.toSAX(calendarColors, new IgnoreRootHandler(contentHandler));
            }
        }
        catch (SourceNotFoundException e)
        {
            // Silently ignore
        }
        finally
        {
            _sourceResolver.release(calendarColors);
        }
    }

    private void _saxAllColors() throws SAXException, MalformedURLException, IOException, ProcessingException
    {
        Source calendarColors = null;
        try
        {
            calendarColors = _sourceResolver.resolveURI("plugin:workspaces://source-calendar-colors.xml");
            if (calendarColors.exists())
            {
                XMLUtils.startElement(contentHandler, "allColors");
                SourceUtil.toSAX(calendarColors, new IgnoreRootHandler(contentHandler));
                XMLUtils.endElement(contentHandler, "allColors");
            }
        }
        catch (SourceNotFoundException e)
        {
            // Silently ignore
        }
        finally
        {
            _sourceResolver.release(calendarColors);
        }
    }

    private void _saxResourceCalendar() throws SAXException
    {
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        
        Calendar calendar = calendarModule.getResourceCalendar(_getProject());

        XMLUtils.startElement(contentHandler, "resourceCalendar");

        XMLUtils.createElement(contentHandler, "id", calendar.getId());
        XMLUtils.createElement(contentHandler, "title", calendar.getName());
        XMLUtils.createElement(contentHandler, "color", calendar.getColor());
        
        XMLUtils.endElement(contentHandler, "resourceCalendar");
    }

    /**
     * Returns the current project
     * @return the current project
     */
    private Project _getProject()
    {
        Project project = null;
        
        Request request = ContextHelper.getRequest(_context);
        String siteName = (String) request.getAttribute("site");
        
        if (StringUtils.isNotEmpty(siteName) && _siteManager.hasSite(siteName))
        {
            Site site = _siteManager.getSite(siteName);
            List<Project> projects = _projectManager.getProjectsForSite(site);
            
            project = !projects.isEmpty() ? projects.get(0) : null;
        }
        
        return project;
    }
    
    private void _saxRecurrenceTypes() throws SAXException
    {
        XMLUtils.startElement(contentHandler, "recurrenceTypes");

        for (EventRecurrenceTypeEnum eventRecurrenceTypeEnum : EventRecurrenceTypeEnum.values())
        {
            XMLUtils.startElement(contentHandler, "recurrenceType");
            
            XMLUtils.createElement(contentHandler, "recurrenceTypeName", eventRecurrenceTypeEnum.name());
            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_EVENT_LIST_PERIOD_" + eventRecurrenceTypeEnum.name()).toSAX(contentHandler, "recurrenceTypeLabel");

            XMLUtils.endElement(contentHandler, "recurrenceType");
        }

        XMLUtils.endElement(contentHandler, "recurrenceTypes");
    }

    private void _saxGlyphs() throws SAXException
    {

        Collection<List<String>> glyphs = new LinkedHashSet<>();
        
        XMLUtils.startElement(contentHandler, "glyphs");
        boolean havePremium = PluginsManager.getInstance().getPluginNames().contains("fontawesome");
        String cssUri = "plugin:fontawesome6" + (havePremium ? "" : "-free") + "://resources/css/";

        glyphs.addAll(_cssFontHelper.getGlyphClassNames(cssUri + "ametys-solid.css", null));
        glyphs.addAll(_cssFontHelper.getGlyphClassNames(cssUri + "ametys-brands.css", null));
        if (havePremium)
        {
            glyphs.addAll(_cssFontHelper.getGlyphClassNames(cssUri + "ametys-light.css", null));
        }
        glyphs.addAll(_cssFontHelper.getGlyphClassNames(cssUri + "ametys-regular.css", null));

        for (List<String> similairGlyphs : glyphs)
        {
            XMLUtils.startElement(contentHandler, "glyph");
            for (String glyph : similairGlyphs)
            {
                XMLUtils.createElement(contentHandler, "alt", glyph);
            }
            XMLUtils.endElement(contentHandler, "glyph");
        }
        XMLUtils.endElement(contentHandler, "glyphs");
    }
}
