/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.modules;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Manager for the Workspaces modules
 */
public interface WorkspaceModule extends Comparable<WorkspaceModule>
{
    /** Order of module */
    static int ORDER_WALLCONTENT = 0;
    /** Order of module */
    static int ORDER_NEWS = 5;
    /** Order of module */
    static int ORDER_ALERTS = 5;
    /** Order of module */
    static int ORDER_MEMBERS = 10;
    /** Order of module */
    static int ORDER_DOCUMENTS = 20;
    /** Order of module */
    static int ORDER_FORUMS = 30;
    /** Order of module */
    static int ORDER_CALENDAR = 40;
    /** Order of module */
    static int ORDER_TASKS = 50;
    /** Order of module */
    static int ORDER_MINISITE = 60;
    /** Order of module */
    static int ORDER_ABOUT = 70;
    /** Header id for elements group header */
    static String GROUP_HEADER_ELEMENTS_ID = "general$elements";
    /** Header id for size group header */
    static String GROUP_HEADER_SIZE_ID = "general$size";
    /** Header id for activated group header */
    static String GROUP_HEADER_ACTIVATED_ID = "general$activated";
    /** Header id for last activity group header */
    static String GROUP_HEADER_LAST_ACTIVITY_ID = "general$lastActivity";

    /**
     * Get the module Id
     * @return The module ID
     */
    String getId();
    
    /**
     * Get the module title
     * @return The title
     */
    I18nizableText getModuleTitle();
    
    /**
     * Get the module title
     * @return The title
     */
    I18nizableText getModuleDescription();
    
    /**
     * Get the module name
     * @return the module name
     */
    String getModuleName();
    
    /**
     * Get the module order
     * @return the priority in menus
     */
    int getOrder();
    
    /**
     * Activate the module for the project, creating the module pages
     * @param project The project
     * @param additionalValues A list of optional additional values. Accepted values are : description, mailingList, inscriptionStatus, defaultProfile, tags, categoryTags, keywords and language
     */
    void activateModule(Project project, Map<String, Object> additionalValues);    
    /**
     * Initialize the sitemap for the module
     * @param project The project of the module
     * @param sitemap The sitemap
     */
    void initializeSitemap(Project project, Sitemap sitemap);
    
    /**
     * Deactivate a module
     * @param project The project
     */
    void deactivateModule(Project project);
    
    /**
     * Delete a module and all related data (pages, events, resources, ...)
     * @param project The project
     */
    void deleteData(Project project);

    /**
     * Get the set of allowed event type for the module
     * @return The set of allowed event types
     */
    Set<String> getAllowedEventTypes();
    
    /**
     * Get the set of all event type for the module
     * @return The set of all event types
     */
    Set<String> getAllEventTypes();
    
    /**
     * Get the module root node for the given project
     * @param project The project containing the module
     * @param create True to create the node if it does not exists
     * @return The root node, or null if it does not exists and was not created
     */
    ModifiableResourceCollection getModuleRoot(Project project, boolean create);

    /**
     * Get the URL of the first module's page
     * @param project The project
     * @return the module url
     */
    String getModuleUrl(Project project);
    
    /**
     * Determines if the module is unactivated by default
     * @return true if the module is not proposed by default
     */
    default boolean isUnactivatedByDefault()
    {
        return false;
    }

    default int compareTo(WorkspaceModule that)
    {
        if (this.getOrder() == that.getOrder())
        {
            return this.getId().compareTo(that.getId());
        }
        else
        {
            return this.getOrder() - that.getOrder();
        }
    }

    /**
     * Get the statistics of the module
     * @param project The project
     * @return a map of statistics
     */
    Map<String, Object> getStatistics(Project project);

    /**
     * Get the headers of statistics
     * @return a list of statistics headers
     */
    List<StatisticColumn> getStatisticModel();
    
    /**
     * Returns the key to get module size in statistics
     * @return The key to get module size in statistics
     */
    public String getModuleSizeKey();
}
