/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.notification;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.observation.Event;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Observer to send mail notifications on workspace member removal
 */
public class RemoveMemberMailNotifierObserver extends AbstractMemberMailNotifierObserver
{
    @Override
    public boolean supports(Event event)
    {
        if (ObservationConstants.EVENT_MEMBER_DELETED.equals(event.getId()))
        {
            return Config.getInstance().getValue("workspaces.member.removed.send.notification", true, false);
        }
        return false;
    }
    
    @Override
    protected I18nizableText getI18nSubject(Event event, Project project)
    {
        return new I18nizableText("plugin." + _pluginName, "PROJECT_MAIL_NOTIFICATION_EVENT_SUBJECT_MEMBER_DELETED", List.of(project.getTitle()));
    }
    
    @Override
    protected String getMailBodyURI(Event event, Project project)
    {
        return "cocoon://_plugins/workspaces/notification-mail-member-event";
    }

    @Override
    protected Map<String, List<String>> getUserToNotifyByLanguage(Event event, Project project)
    {
        // Retrieve removed member
        Map<String, Object> args = event.getArguments();
        String identity = (String) args.get(ObservationConstants.ARGS_MEMBER_IDENTITY);
        MemberType identityType = (MemberType) args.get(ObservationConstants.ARGS_MEMBER_IDENTITY_TYPE);
        
        if (MemberType.USER == identityType)
        {
            return Optional.of(identity)
                    .map(UserIdentity::stringToUserIdentity)
                    .filter(userIdentity -> !_projectMemberManager.isProjectMember(project, userIdentity)) // user is not in a group that is still part of the project
                    .map(_userManager::getUser)
                    .map(user -> Pair.of(user, user.getEmail()))
                    .filter(p -> StringUtils.isNotEmpty(p.getRight()))
                    .map(p -> Map.of(StringUtils.defaultString(p.getLeft().getLanguage()), List.of(p.getRight())))
                    .orElse(Map.of());
        }
        else if (MemberType.GROUP == identityType)
        {
            GroupIdentity groupIdentity = GroupIdentity.stringToGroupIdentity(identity);
            if (groupIdentity != null)
            {
                Group group = _groupManager.getGroup(groupIdentity);
                if (group != null)
                {
                    return group.getUsers().stream()
                        .filter(u -> !_projectMemberManager.isProjectMember(project, u))
                        .map(_userManager::getUser)
                        .map(user -> Pair.of(user, user.getEmail()))
                        .filter(p -> StringUtils.isNotEmpty(p.getRight()))
                        .collect(Collectors.groupingBy(
                                p -> {
                                    return StringUtils.defaultString(p.getLeft().getLanguage());
                                },
                                Collectors.mapping(
                                        Pair::getRight,
                                        Collectors.toList()
                                )
                            )
                        );
                }
            }
        }
        return Map.of();
    }
}
