/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.notification;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.AbstractContentFunction;
import org.ametys.core.observation.Event;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Function to mark that a news is notified  
 *
 */
public class SetNotifiedFunction extends AbstractContentFunction
{
    /** Property to mark as notified */
    public static final String NOTIFIED_PROPERTY_NAME = "notified";
    
    /** The user manager */
    protected ContentTypesHelper _cTypeHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _cTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
    }
    
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        WorkflowAwareContent content = getContent(transientVars);
        
        if (_cTypeHelper.isInstanceOf(content, WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID))
        {
            try
            {
                _markAsNotified(content, transientVars);
                
                _createVersion(content);
                
                _addLabel(content, CmsConstants.LIVE_LABEL);
                
                _notifyObservers(content, transientVars);
            }
            catch (RepositoryException e)
            {
                throw new WorkflowException("Unable to set notified markup", e);
            }
        }
    }
    
    /**
     * Mark the content as notified
     * @param content the content
     * @param transientVars the transient variables
     */
    protected void _markAsNotified(WorkflowAwareContent content, Map transientVars)
    {
        if (content != null)
        {
            content.getInternalDataHolder().setValue(NOTIFIED_PROPERTY_NAME, true);
            content.saveChanges();
        }
    }
    
    /**
     * Notify observers of news publication
     * @param content the content 
     * @param transientVars the transient variables
     * @throws WorkflowException if an error occurred
     */
    protected void _notifyObservers(WorkflowAwareContent content, Map transientVars) throws WorkflowException
    {
        if (content != null)
        {
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.cms.ObservationConstants.ARGS_CONTENT, content);
            eventParams.put(org.ametys.cms.ObservationConstants.ARGS_CONTENT_ID, content.getId());
            _observationManager.notify(new Event(ObservationConstants.EVENT_PROJECT_NEWS_PUBLISHED, getUser(transientVars), eventParams));
            
            _observationManager.notify(new Event(org.ametys.cms.ObservationConstants.EVENT_CONTENT_VALIDATED, getUser(transientVars), eventParams));
        }
    }

    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.POST;
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_SET_NOTIFIED_FUNCTION_LABEL");
    }
    
}
