/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.rights.accesscontroller;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.codec.binary.StringUtils;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractRightBasedAccessController;
import org.ametys.plugins.frontedition.AmetysFrontEditionHelper;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.ProjectConstants;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.ProjectManager.UnknownCatalogSiteException;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * {@link AccessController} for a {@link Project}
 * The projects' managers have some rights on their projects
 * The projects' members can read their projects
 */
public class ProjectAccessController extends AbstractRightBasedAccessController implements Serviceable
{
    /** the workspace category */
    public static final I18nizableText WORKSPACE_CONTEXT_CATEGORY = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_ACCESS_CONTROLLER_CONTEXT_CATEGORY");
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The project members */
    protected ProjectMemberManager _projectMembers;
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The rights to give for managers */
    protected Set<String> _managerRights = Set.of(ProjectConstants.RIGHT_PROJECT_EDIT, ProjectConstants.RIGHT_PROJECT_DELETE, AmetysFrontEditionHelper.FRONT_EDITION_RIGHT_ID);
    /** The rights to give for members */
    protected Set<String> _memberRights = Set.of(AmetysFrontEditionHelper.FRONT_EDITION_RIGHT_ID); // Needed to allow front-edition access on home page (to create news or alerts)
    

    public void service(ServiceManager manager) throws ServiceException
    {
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _projectMembers = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    public boolean supports(Object object)
    {
        return object instanceof Project;
    }

    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        Project project = (Project) object;
        
        if (_managerRights.contains(rightId)
            && ArrayUtils.contains(project.getManagers(), user))
        {
            return AccessResult.USER_ALLOWED;
        }
        else if (_memberRights.contains(rightId))
        {
            ProjectMember projectMember = _projectMembers.getProjectMember(project, user, userGroups);
            if (projectMember != null)
            {
                return MemberType.USER == projectMember.getType() ? AccessResult.USER_ALLOWED : AccessResult.GROUP_ALLOWED;
            }
        }
        
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Project project = (Project) object;
        
        if (ArrayUtils.contains(project.getManagers(), user))
        {
            return AccessResult.USER_ALLOWED;
        }
        else
        {
            ProjectMember projectMember = _projectMembers.getProjectMember(project, user, userGroups);
            if (projectMember != null)
            {
                return MemberType.USER == projectMember.getType() ? AccessResult.USER_ALLOWED : AccessResult.GROUP_ALLOWED;
            }
        }
        
        return AccessResult.UNKNOWN;
    }

    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Map<String, AccessResult> permissionByRight = new HashMap<>();
        
        Project project = (Project) object;
        if (ArrayUtils.contains(project.getManagers(), user))
        {
            for (String managerRight : _managerRights)
            {
                permissionByRight.put(managerRight, AccessResult.USER_ALLOWED);
            }
        }
        else
        {
            ProjectMember projectMember = _projectMembers.getProjectMember(project, user, userGroups);
            if (projectMember != null)
            {
                for (String memberRight : _memberRights)
                {
                    permissionByRight.put(memberRight, MemberType.USER == projectMember.getType() ? AccessResult.USER_ALLOWED : AccessResult.GROUP_ALLOWED);
                }
            }
        }
        
        return permissionByRight;
    }
    
    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        Map<UserIdentity, AccessResult> permissionByUser = new HashMap<>();
        
        if (_managerRights.contains(rightId))
        {
            Project project = (Project) object;
            
            for (UserIdentity manager : project.getManagers())
            {
                permissionByUser.put(manager, AccessResult.USER_ALLOWED);
            }
        }
        
        return permissionByUser;
    }
    
    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        Map<UserIdentity, AccessResult> permissionByUser = new HashMap<>();
        
        Project project = (Project) object;
        
        for (UserIdentity manager : project.getManagers())
        {
            permissionByUser.put(manager, AccessResult.USER_ALLOWED);
        }
        for (ProjectMember member : _projectMembers.getProjectMembers(project, false))
        {
            if (member.getType() == MemberType.USER)
            {
                permissionByUser.put(member.getUser().getIdentity(), AccessResult.USER_ALLOWED);
            }
        }
        
        return permissionByUser;
    }
    
    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Map.of();
    }
    
    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        Map<GroupIdentity, AccessResult> permissionByGroup = new HashMap<>();
        
        Project project = (Project) object;
        
        for (ProjectMember member : _projectMembers.getProjectMembers(project, false))
        {
            if (member.getType() == MemberType.GROUP)
            {
                permissionByGroup.put(member.getGroup().getIdentity(), AccessResult.USER_ALLOWED);
            }
        }
        
        return permissionByGroup;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }
    
    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }
    
    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }
    
    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }
    
    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    protected AccessExplanation _getAccessExplanation(AccessResult result, Object object, UserIdentity user, Set<GroupIdentity> groups, String rightId)
    {
        switch (result)
        {
            case USER_ALLOWED:
                Project project = (Project) object;
                if ((rightId == null || _managerRights.contains(rightId))
                        && ArrayUtils.contains(project.getManagers(), user))
                {
                    return new AccessExplanation(
                            getId(),
                            result,
                            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_ACCESS_CONTROLLER_MANAGER_EXPLANATION", Map.of("title", new I18nizableText(project.getTitle())))
                            );
                }
                else
                {
                    return new AccessExplanation(
                            getId(),
                            AccessResult.USER_ALLOWED,
                            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_ACCESS_CONTROLLER_USER_EXPLANATION", Map.of("title", new I18nizableText(project.getTitle())))
                            );
                }
            case GROUP_ALLOWED:
                project = (Project) object;
                ProjectMember projectMember = _projectMembers.getProjectMember(project, user, groups);
                return new AccessExplanation(
                        getId(),
                        AccessResult.GROUP_ALLOWED,
                        new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_ACCESS_CONTROLLER_GROUP_EXPLANATION",
                                Map.of("title", new I18nizableText(project.getTitle()), "group", new I18nizableText(projectMember.getGroup().getLabel())))
                        );
            case UNKNOWN:
                project = (Project) object;
                return new AccessExplanation(
                        getId(),
                        result,
                        new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_ACCESS_CONTROLLER_UNKNOWN_EXPLANATION", Map.of("title", new I18nizableText(project.getTitle())))
                    );
            default:
                return AccessController.getDefaultAccessExplanation(getId(), result);
        }
    }

    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Project project)
        {
            return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_ACCESS_CONTROLLER_OBJECT_LABEL", Map.of("title", new I18nizableText(project.getTitle())));
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }
    
    public I18nizableText getObjectCategory(Object object)
    {
        return WORKSPACE_CONTEXT_CATEGORY;
    }
    
    public int getObjectPriority(Object object)
    {
        return 10;
    }

    @Override
    protected Iterable< ? extends Object> getHandledObjects(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        Site site = _siteManager.getSite(siteName);
        if (site != null)
        {
            try
            {
                if (StringUtils.equals(siteName, _projectManager.getCatalogSiteName()))
                {
                    return _projectManager.getUserProjects(identity).keySet();
                }
                else
                {
                    return _projectManager.getProjectsForSite(site);
                }
            }
            catch (UnknownCatalogSiteException e)
            {
                // Ignore the no catalog sitename exception
            }
        }
        return List.of();
    }
    
    @Override
    protected Collection<String> getHandledRights()
    {
        HashSet<String> rights = new HashSet<>(_managerRights);
        rights.addAll(_memberRights);
        return rights;
    }
}
