/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Request;
import org.xml.sax.SAXException;

import org.ametys.cms.search.SearchResults;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;

/**
 * Abstract generator for search module of type content
 *
 */
public abstract class AbstractSolrSearchModuleGenerator extends AbstractSearchModuleGenerator
{
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
    }
    
    @Override
    protected void saxHits(String siteName, String lang, String textfield, Request request, int offset, int limit, int minLimit) throws SAXException, ProcessingException
    {
        try
        {
            SearchResults<? extends AmetysObject> searchResults = getSearchResults(siteName, lang, textfield, request, offset, limit);
            
            long totalCount = searchResults.getTotalCount();
            AmetysObjectIterable<? extends AmetysObject> results = searchResults.getObjects();
            List< ? extends AmetysObject> resultList = results.stream().collect(Collectors.toList());
            
            saxHits(resultList, lang, offset, limit, minLimit, totalCount);
        }
        catch (Exception e)
        {
            throw new ProcessingException("Unable to get search results", e);
        }
    }

    /**
     * Get the search results
     * @param siteName the current site name
     * @param lang the current language
     * @param textfield the search input
     * @param request the request
     * @param offset the search offset
     * @param limit the max number of results
     * @return the search results
     * @throws Exception if the search failed
     */
    protected abstract SearchResults<? extends AmetysObject> getSearchResults(String siteName, String lang, String textfield, Request request, int offset, int limit) throws Exception;
    
}
