/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.FullTextExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.CalendarColorsComponent;
import org.ametys.plugins.workspaces.calendars.CalendarColorsComponent.CalendarColor;
import org.ametys.plugins.workspaces.calendars.CalendarWorkspaceModule;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventOccurrence;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Generator for calendar events search module
 */
public class EventSearchModuleGenerator extends AbstractXpathSearchModuleGenerator
{
    /** Calendar module */
    protected CalendarWorkspaceModule _calendarModule;
    /** The color component */
    protected CalendarColorsComponent _calendarColors;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _calendarModule = moduleManagerEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        _calendarColors = (CalendarColorsComponent) smanager.lookup(CalendarColorsComponent.ROLE);
    }
    
    @Override
    protected String getXPathQuery(String siteName, String lang, String textfield, Request request, int offset, int limit) throws Exception
    {
        List<Project> projects = getProjects(request, true);
        
        List<Project> filteredProjects = filterProjectsForModule(projects, CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        if (filteredProjects.isEmpty())
        {
            // No project available for this module
            return null;
        }
        
        String projectXPathQuery = getProjectXPathQuery(filteredProjects);
        
        String searchQuery = "";
        if (StringUtils.isNotBlank(textfield))
        {
            Expression titleExpr = new FullTextExpression("title", textfield);
            Expression descExpr = new FullTextExpression("description", textfield);
            
            OrExpression expr = new OrExpression(titleExpr, descExpr);
            searchQuery = "[" + expr.build() + "]";
        }
        
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addCriterion("startDate", false, false);
        String sortQuery = sortCriteria.build();
        
        String jcrQuery = projectXPathQuery + "//element(*, ametys:calendar-event)" + searchQuery + " " + sortQuery;
        return jcrQuery;
    }

    @Override
    protected void saxHit(AmetysObject object, String lang) throws Exception
    {
        if (object instanceof CalendarEvent)
        {
            CalendarEvent calendarEvent = (CalendarEvent) object;
            Project project = getProject(object);
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", calendarEvent.getId());
            XMLUtils.startElement(contentHandler, "hit", attrs);
            
            String eventUri = _calendarModule.getEventUri(project, calendarEvent.getParent().getId(), calendarEvent.getId());
            XMLUtils.createElement(contentHandler, "uri", eventUri);
            
            XMLUtils.createElement(contentHandler, "title", calendarEvent.getTitle());
            
            XMLUtils.createElement(contentHandler, "fullDay", String.valueOf(calendarEvent.getFullDay()));
            XMLUtils.createElement(contentHandler, "creationDate", DateUtils.zonedDateTimeToString(calendarEvent.getCreationDate()));
            XMLUtils.createElement(contentHandler, "startDate", DateUtils.zonedDateTimeToString(calendarEvent.getStartDate()));
            XMLUtils.createElement(contentHandler, "endDate", DateUtils.zonedDateTimeToString(calendarEvent.getEndDate()));
            XMLUtils.createElement(contentHandler, "recurrenceType", calendarEvent.getRecurrenceType().name().toLowerCase());
            Optional<CalendarEventOccurrence> nextOccurrence = calendarEvent.getNextOccurrence(new CalendarEventOccurrence(calendarEvent, ZonedDateTime.now()));
            if (nextOccurrence.isPresent())
            {
                XMLUtils.createElement(contentHandler, "nextOccurence", DateUtils.zonedDateTimeToString(nextOccurrence.get().getStartDate()));
            }
            if (calendarEvent.getDescription() != null)
            {
                XMLUtils.createElement(contentHandler, "description", calendarEvent.getDescription());
            }
            if (calendarEvent.getLocation() != null)
            {
                XMLUtils.createElement(contentHandler, "location", calendarEvent.getLocation());
            }
            
            saxUser(calendarEvent.getCreator(), "creator");
            
            saxProject(project);
            
            saxCalendar(getCalendar(calendarEvent));
            
            XMLUtils.endElement(contentHandler, "hit");
        }
    }
    
    /**
     * Get the parent calendar
     * @param event the event
     * @return the parent calendar or null if not found
     */
    protected Calendar getCalendar(CalendarEvent event)
    {
        AmetysObject parent = event.getParent();
        while (parent != null)
        {
            if (parent instanceof Calendar)
            {
                return (Calendar) parent;
            }
            parent = parent.getParent();
        }
        
        return null;
    }
    
    /**
     * SAX the calendar
     * @param calendar the calendar
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxCalendar(Calendar calendar) throws SAXException
    {
        if (calendar != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", calendar.getId());
            XMLUtils.startElement(contentHandler, "calendar", attrs);
            XMLUtils.createElement(contentHandler, "title", calendar.getName());
            
            CalendarColor color = _calendarColors.getColor(calendar.getColor());
            if (color != null)
            {
                XMLUtils.createElement(contentHandler, "color", color.getColor());
                XMLUtils.createElement(contentHandler, "textColor", color.getTextColor());
            }
            
            XMLUtils.endElement(contentHandler, "calendar");
        }
    }

}
