/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.query;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.QuerySyntaxException;
import org.ametys.plugins.workspaces.indexing.solr.SolrWorkspacesConstants;

/**
 * Represents a {@link Query} testing the project property.
 */
public class ProjectQuery implements Query
{
    private Operator _operator;
    private List<String> _ids;
    
    /**
     * Build a ProjectQuery to test if the project property exits
     */
    public ProjectQuery()
    {
        this(Operator.EQ, "*");
    }
    
    /**
     * Build a ProjectQuery to test if the project property is equal to one of the given project ids
     * @param ids the project ids.
     */
    public ProjectQuery(String... ids)
    {
        this(Operator.EQ, ids);
    }
    
    /**
     * Build a ProjectQuery to test if the project property is equal to one of the given project ids
     * @param ids the project ids.
     */
    public ProjectQuery(Collection<String> ids)
    {
        this(Operator.EQ, ids);
    }
    
    /**
     * Build a ProjectQuery to test if the project property is equal or different to one of the given project ids
     * @param operator the operator (equal ot not-equal)
     * @param ids the project ids.
     */
    public ProjectQuery(Operator operator, String... ids)
    {
        this(operator, Arrays.asList(ids));
    }
    
    /**
     * Build a ProjectQuery.
     * @param operator the operator.
     * @param ids the project ids.
     */
    public ProjectQuery(Operator operator, Collection<String> ids)
    {
        if (Operator.EQ != operator && Operator.NE != operator)
        {
            throw new IllegalArgumentException("Test operator '" + operator + "' is unknown for test's expression.");
        }
        
        _operator = operator;
        _ids = new ArrayList<>(ids);
    }
    
    @Override
    public String build() throws QuerySyntaxException
    {
        int count = _ids.size();
        
        StringBuilder sb = new StringBuilder();
        
        if (_operator == Operator.NE)
        {
            sb.append('-');
        }
        if (count > 1)
        {
            sb.append('(');
        }
        
        boolean first = true;
        for (String id : _ids)
        {
            if (!first)
            {
                sb.append(" OR ");
            }
            sb.append(SolrWorkspacesConstants.PROJECT_ID).append(":\"").append(id).append("\"");
            first = false;
        }
        
        if (count > 1)
        {
            sb.append(')');
        }
        
        return sb.toString();
    }
}
