/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tags;

import java.util.Collection;
import java.util.Map;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.tag.DefaultTag;
import org.ametys.cms.tag.jcr.AbstractJCRTagProvider;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Class representing a project jcr tag provider. <br>
 */
public class ProjectJCRTagProvider extends AbstractJCRTagProvider<DefaultTag>
{
    /** Constant for plugin node name */ 
    public static final String PROJECT_PLUGIN_NODE_NAME = "workspaces";
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The projet tag provider extension point */
    protected ProjectTagProviderExtensionPoint _projectTagProviderEP;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _projectTagProviderEP = (ProjectTagProviderExtensionPoint) smanager.lookup(ProjectTagProviderExtensionPoint.ROLE);
    }
    
    @Override
    protected void _fillCache(TraversableAmetysObject parentTagNode, DefaultTag parentTag, Map<String, DefaultTag> cache) throws RepositoryException
    {
        for (AmetysObject child : parentTagNode.getChildren())
        {
            if (child instanceof ProjectTagJCR)
            {
                ProjectTagJCR jcrTag = (ProjectTagJCR) child;
                DefaultTag tag = new DefaultTag(jcrTag.getId(), jcrTag.getName(), parentTag, new I18nizableText(jcrTag.getTitle()), new I18nizableText(jcrTag.getDescription()));
                cache.put(child.getName(), tag);
                
                if (parentTag != null)
                {
                    parentTag.addTag(tag);
                }
                
                _fillCache(jcrTag, tag, cache);
            }
        }
    }
    
    @Override
    protected ModifiableTraversableAmetysObject _getRootNode(Map<String, Object> contextualParameters, Session session) throws RepositoryException
    {
        try
        {
            ModifiableTraversableAmetysObject pluginsNode = _resolver.resolveByPath("/ametys:plugins", session);
            
            ModifiableTraversableAmetysObject pluginNode = _getOrCreateNode(pluginsNode, PROJECT_PLUGIN_NODE_NAME, "ametys:unstructured");
            
            return _getOrCreateNode(pluginNode, _projectTagProviderEP.getTagsNodeName(), _projectTagProviderEP.getTagsNodeType());
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the JCR tags root node", e);
        }
    }
    
    @Override
    public Collection<DefaultTag> getTags(String tagID, Map<String, Object> contextualParameters)
    {
        DefaultTag tag = getTag(tagID, contextualParameters);
        return tag != null ? tag.getTags().values() : null;
    }
}
