/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks.actions;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.tasks.Task;

/**
 * Action providing data about Tasks.
 */
public abstract class AbstractGetTasksForExportAction extends ServiceableAction
{
    private static final String __FULL_DAY_PATTERN = "uuuuMMdd";
    
    /** The object resolver */
    protected AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);
        List<Task> tasks = getTasks(request);
        
        List<Map<String, Object>> tasksMap = tasks.stream()
                .filter(this::_passFilter)
                .map(this::_task2Result).toList();
        
        result.put("events", tasksMap);
        
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        return EMPTY_MAP;
    }
    
    /**
     * Check if a given task match the requirements to be parsed
     * @param task the task
     * @return true if the task match the requirements
     */
    protected boolean _passFilter(Task task)
    {
        return task != null;
    }
    
    /**
     * Get the task map
     * @param task the task
     * @return the task map
     */
    protected Map<String, Object> _task2Result(Task task)
    {
        Map<String, Object> taskMap = new HashMap<>();
        taskMap.put("id", task.getId());
        taskMap.put("title", task.getLabel());
        taskMap.put("description", task.getDescription());
        taskMap.put("fullDay", true);
        taskMap.put("keywords", task.getTags());
        LocalDate dueDate = task.getDueDate();
        LocalDate startDate = task.getStartDate();
        LocalDate startDateEvent = startDate != null ? startDate : dueDate;
        if (startDateEvent != null)
        {
            taskMap.put("startDate", startDateEvent.format(DateTimeFormatter.BASIC_ISO_DATE));
        }
        if (dueDate != null)
        {
            taskMap.put("endDate", dueDate.plusDays(1).format(DateTimeFormatter.BASIC_ISO_DATE));
        }
        taskMap.put("creationDate", DateUtils.zonedDateTimeToString(task.getCreationDate(), task.getCreationDate().getZone(), __FULL_DAY_PATTERN));
        taskMap.put("lastModified", DateUtils.zonedDateTimeToString(task.getLastModified(), task.getLastModified().getZone(), __FULL_DAY_PATTERN));
        taskMap.put("tags", task.getTags());
        taskMap.put("recurrenceType", "NEVER");
        return taskMap;
    }
    
    /**
     * Get the list of tasks
     * @param request the request.
     * @return the list of tasks
     */
    protected abstract List<Task> getTasks(Request request);
}
