/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * This class represents a result column
 */
public class StatisticColumn
{

    private String _id;

    private I18nizableText _label;
    
    private StatisticsColumnType _type;

    private Integer _width;

    private String _renderer;

    private String _converter;

    private Boolean _hidden;

    private Boolean _editable;

    private Boolean _sortable;

    private Boolean _multiple;

    private Boolean _filter;

    private String _defaultSorter;

    private List<StatisticColumn> _subColumns;

    private String _group;

    /**
     * Create a StatisticColumn
     * @param id the id of StatisticColumn
     * @param label the label of StatisticColumn
     */
    public StatisticColumn(String id, I18nizableText label)
    {
        this._id = id;
        this._label = label;
        this._subColumns = new ArrayList<>();
        this._filter = true;
    }
        
    /**
     * Set the id
     * @param id the id
     * @return StatisticColumn with a set id
     */
    public StatisticColumn withId(String id)
    {
        this._id = id;
        return this;
    }

    /**
     * Set the label
     * @param label the label
     * @return StatisticColumn with a set label
     */
    public StatisticColumn withLabel(I18nizableText label)
    {
        this._label = label;
        return this;
    }
    
    /**
     * Set the type
     * @param type the type
     * @return StatisticColumn with a set type
     */
    public StatisticColumn withType(StatisticsColumnType type)
    {
        this._type = type;
        return this;
    }

    /**
     * Set the width
     * @param width the width
     * @return StatisticColumn with a set width
     */
    public StatisticColumn withWidth(int width)
    {
        this._width = width;
        return this;
    }

    /**
     * Set the renderer
     * @param renderer the renderer
     * @return StatisticColumn with a set renderer
     */
    public StatisticColumn withRenderer(String renderer)
    {
        this._renderer = renderer;
        return this;
    }

    /**
     * Set the converter
     * @param converter the converter
     * @return StatisticColumn with a set converter
     */
    public StatisticColumn withConverter(String converter)
    {
        this._converter = converter;
        return this;
    }

    /**
     * Set the hidden boolean
     * @param hidden is the column hidden
     * @return StatisticColumn with a set hidden value
     */
    public StatisticColumn isHidden(boolean hidden)
    {
        this._hidden = hidden;
        return this;
    }

    /**
     * Set the editable boolean
     * @param editable is the column editable
     * @return StatisticColumn with a set editable value
     */
    public StatisticColumn withEditable(boolean editable)
    {
        this._editable = editable;
        return this;
    }

    /**
     * Set the sortable boolean
     * @param sortable is the column sortable
     * @return StatisticColumn with a set sortable value
     */
    public StatisticColumn withSortable(boolean sortable)
    {
        this._sortable = sortable;
        return this;
    }

    /**
     * Set the multiple boolean
     * @param multiple is the column multiple
     * @return StatisticColumn with a set multiple value
     */
    public StatisticColumn withMultiple(boolean multiple)
    {
        this._multiple = multiple;
        return this;
    }

    /**
     * Set the defaultSorter
     * @param defaultSorter the defaultSorter
     * @return StatisticColumn with a set defaultSorter
     */
    public StatisticColumn withDefaultSorter(String defaultSorter)
    {
        this._defaultSorter = defaultSorter;
        return this;
    }

    /**
     * Set the group
     * @param group the group
     * @return StatisticColumn with a set group
     */
    public StatisticColumn withGroup(String group)
    {
        this._group = group;
        return this;
    }

    /**
     * Set the filter boolean
     * @param filter is the column filterable
     * @return StatisticColumn with a set filter value
     */
    public StatisticColumn withFilter(boolean filter)
    {
        this._filter = filter;
        return this;
    }
    
    /**
     * Set the columns
     * @param subColumn the column
     * @return StatisticColumn with a set columns
     */
    public StatisticColumn addSubColumn(StatisticColumn subColumn)
    {
        this._subColumns.add(subColumn);
        return this;
    }
    
    /**
     * Set the columns
     * @param subColumns the columns
     * @return StatisticColumn with a set columns
     */
    public StatisticColumn addSubColumns(List<StatisticColumn> subColumns)
    {
        this._subColumns.addAll(subColumns);
        return this;
    }

    /**
     * Get the group of the column
     * @return the group of the column
     */
    public String getGroup()
    {
        return _group;
    }
    
    /**
     * Get the id of the column
     * @return the id of the column
     */
    public String getId()
    {
        return _id;
    }
    
    /**
     * Get the label
     * @return The label
     */
    public I18nizableText getLabel()
    {
        return _label;
    }

    /**
     * convert the StatisticColumn into a map
     * @return the map of StatisticColumn attributes
     */
    public Map<String, Object> convertToJSON()
    {
        Map<String, Object> json = new HashMap<>();
        
        json.put("id", _id);
        json.put("path", _id);
        json.put("label", _label);
        json.put("type", _type != null ? _type.getId().toLowerCase() : StringUtils.EMPTY);
        json.put("width", _width);
        json.put("renderer", _renderer);
        json.put("converter", _converter);
        json.put("hidden", _hidden);
        json.put("editable", _editable);
        json.put("sortable", _sortable);
        json.put("multiple", _multiple);
        json.put("filter", _filter);
        json.put("defaultSorter", _defaultSorter);

        if (!this._subColumns.isEmpty())
        {
            List<Object> jsonObject = new ArrayList<>();
            
            for (StatisticColumn column : _subColumns)
            {
                jsonObject.add(column.convertToJSON());
            }
            
            json.put("subColumns", jsonObject);
        }

        return json;
    }
}
