/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
 WallContent = {
    
    /**
     * @property {RegExp} Regular expression to extract link from plain text
     */
    _linkRe: /([\s]|^)(((https?\:\/\/)|(www\.))(\S+))/gi,
    // _linkRe: /([\s]|^)(((https?\:\/\/)|(www\.))(\S+)(?<![.,;:?!]))/gi, // ES2018 only
    
    /**
     * Function to refresh the wall without reloading the page
     */
    refresh: null,
    
    /**
     * Reload the page
     * @param {String} [contentId] If not null, go to content with anchor
     */
    reload: function(contentId)
    {
        var anchor = "";
        if (contentId)
        {
            var i = contentId.indexOf('://');
	        anchor = '#content-' + contentId.substring(i + 3);
        }
        
        /([^?#]*)([\?#].*)*/.test(document.location.href);
        document.location.href = RegExp.$1 + "?foo=" + Math.random() + anchor; 
    },
    
    /**
	  * Publish a new wall content
	  * @param {Button} btn the publish button
	  * @param {Function} [callback] the callback to invoked in case of success
	  */
	 publish: function(btn, callback)
	 {
	    // Disabled button
        this._disableBtn(btn);
        this._loadingBtn(btn);
	    
	    var form = $j(btn).closest('form');
        
        var contentInput = form.find('textarea[name=content]');
        var content = this._convertTextToRichText(contentInput.val(), true);
        
        var imgInput = form.find('input[type=file]')[0];
        var file = imgInput.files.length > 0 ? imgInput.files[0] : null;
        
	    AmetysFront.ServerComm.callMethod({
	        role: 'org.ametys.plugins.workspaces.wall.WallContentManager',
	        methodName: 'publishContent',
	        parameters: [content, file, AmetysFront.getAppParameter('siteName'), AmetysFront.getAppParameter('sitemapLanguage')],
	        callback: {
	            handler: this._publishCb,
                errorHandler:this._publishCb,
	            scope: this,
	            arguments: [callback, btn]
	        }
	    });
	    
	    return false;
	 },
     
     _publishCb: function (data, args)
	 {
	    if (data && data.success)
	    {
            var btn = args[1];
            $j(btn).removeClass("loading");
            
            // Reset form
            var form = $j(btn).closest('form');
            form[0].reset();
            form.find('input[type=file]').trigger('change');
            
	        var callback = args[0];
	        if (typeof (callback) == 'function')
	        {
	            callback(data);
	        }
	        else
	        {        
                if (this.refresh)
                {
                    this.refresh();
                    this._stopLoadingBtn(btn);
                }
                else
                {
                    this.reload(data.contentId);
                }
	        }
	    }
	    else
	    {
            var btn = args[1];
            this._enableBtn(btn);
            this._stopLoadingBtn(btn);
            
	        AmetysFront.Utils.error(data || "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_SUBMIT_ERROR}}");
	    }
	 },
     
    /**
	 * Delete a wall content
	 * @param {String} contentId The content id to delete
	*/
    deleteContent: function(contentId)
	{
        var me = this;
	    AmetysFront.Utils.confirm("{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_DELETE_TITLE}}", "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_DELETE_CONFIRM}}", function () {
	       AmetysFrontEdition.deleteContent(contentId, me._deleteCb, true);
        });
	    
	    return false;
	},
    
    _deleteCb: function(contents)
	{
	    if (contents && contents.length > 0)
	    {
            // Remove HTML if found
            var found = false;
            $j(contents).each(function(i, content) {
                var contentEl = $j('[data-content-id="' + content.id + '"]');
                if (contentEl.length != 0)
                {
                    contentEl.parent().remove();
                    found = true;
                }
            });
            
            if (!found)
            {
                // No element found, reload the wall
                WallContent.reload();
            }
	    }
	    else
	    {
	        AmetysFront.Utils.error("{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_DELETE_ERROR}}");
	    }
	},
    
    /**
	 * Edit a wall content
	 * @param {String} contentId The content id to edit
	 */
	edit: function(contentId)
	{
        var form = $j('[data-content-id="' + contentId + '"]').find('[data-content-edit]');
        
        var content = $j('[data-content-id="' + contentId + '"] [data-ametys-wallcontent]');
        var text = this._convertRichTextToText(content.html());
            
        form.find('textarea').val(text);
        form.find('textarea').trigger('focus');
        form.show();
        
        $j('[data-content-id="' + contentId + '"] [data-content-wrapper]').hide();
	},
    
    /**
     * Cancel changes
     */
    cancelChanges: function (contentId)
	{
        var form = $j('[data-content-id="' + contentId + '"]').find('[data-content-edit]');
        form.hide();
        
	    $j('[data-content-id="' + contentId + '"] [data-content-wrapper]').show();
	},
    
    /**
     * Save changes
     * @param {String} contentId The content id to save
     * @param {Number} [editActionId] the id of workflow action for edition. Defaults to 2.
     */
    saveChanges: function(contentId, editActionId)
	{
        editActionId = editActionId || 2;
        
        var form = $j('[data-content-id="' + contentId + '"]').find('[data-content-edit]');
        
	    var text = form.find('textarea').val();
	    var content = this._convertTextToRichText(text, true);
	    
        var me = this;
	    AmetysFrontEdition.load(function() {
	        AmetysFrontEdition.comm.save(editActionId, contentId, "content", content, function(response) { me._saveCb(response, contentId)});
	    });
	},
    
    _saveCb: function(response, contentId)
	{
	    if (response)
	    {
            if (WallContent.refresh)
            {
                WallContent.refresh();
            }
            else
            {
                WallContent.reload(contentId);
            }
	    }
	    else
	    {
	        AmetysFront.Utils.error("{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_EDIT_ERROR}}");
	    }
	},
    
    _enableBtn: function(btn)
    {
        $j(btn).removeClass('disabled');
        $j(btn).removeAttr("disabled");
    },
    
    _disableBtn: function(btn)
    {
        $j(btn).addClass('disabled');
        $j(btn).attr("disabled","disabled");
    },
    
    _loadingBtn: function(btn)
    {
        $j(btn).closest('form').find('button').addClass('loading');
    },
    
    _stopLoadingBtn: function(btn)
    {
        $j(btn).closest('form').find('button').removeClass('loading');
    },
    
    /**
     * Pin a wall content
     * @param {String} contentId the content id
     */
    pin: function(contentId)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.workspaces.wall.WallContentManager',
            methodName: 'pinContent',
            parameters: [contentId, AmetysFront.getAppParameters()],
            callback: {
                handler: this._pinCb,
                scope: this,
                arguments: [true]
            }
        });
        return false;
    },
    
    _pinCb: function(response, args)
    {
        if (response && response['allright-contents'] && response['allright-contents'].length > 0)
        {
            if (WallContent.refresh)
            {
                WallContent.refresh();
            }
            else
            {
                WallContent.reload(response['allright-contents'][0].id);
            }
        }
        else
        {
            var error = args[0] ? "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_PIN_ERROR}}" : "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_UNPIN_ERROR}}";
            AmetysFront.Utils.error(error);
        }
    },
    
    /**
     * Unpin a wall content
     * @param {String} contentId the content id
     */
    unpin: function(contentId)
    {
         AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.workspaces.wall.WallContentManager',
            methodName: 'unpinContent',
            parameters: [contentId, AmetysFront.getAppParameters()],
            callback: {
                handler: this._pinCb,
                scope: this,
                arguments: [false]
            }
        });
        return false;
    },
    
    /**
     * Add 'like' reaction on a wall content 
     * @param {String} contentId the content id
     * @param {Function} [callback] the callback function to invoked after reacting to content
     */
    like: function(btn, contentId, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.workspaces.wall.WallContentManager',
            methodName: 'react',
            parameters: [contentId, 'LIKE', false],
            callback: {
                handler: this._likeOrUnlikeCb,
                scope: this,
                arguments: [callback, btn, true]
            }
        });
    },
    
    /**
     * Remove 'like' reaction on a wall content 
     * @param {String} contentId the content id
     * @param {Function} [callback] the callback function to invoked after reacting to content
     */
    unlike: function(btn, contentId, callback)
    {
        AmetysFront.ServerComm.callMethod({
            role: 'org.ametys.plugins.workspaces.wall.WallContentManager',
            methodName: 'react',
            parameters: [contentId, 'LIKE', true],
            callback: {
                handler: this._likeOrUnlikeCb,
                scope: this,
                arguments: [callback, btn, false]
            }
        });
    },
    
    
    _likeOrUnlikeCb: function (result, args)
     {
        if (result)
        {
            if (result.unauthorized)
            {
                AmetysFront.Utils.error("{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_LIKE_UNLIKE_UNAUTHORIZED}}");
            }
            else
            {
                var callback = args[0];
                if (typeof (callback) == 'function')
                {
                    callback(args[1], result, args[2]);
                }
                else
                {            
                    this.reload(result.contentId);
                }
            }
        }
        else
        {
            AmetysFront.Utils.error("{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_LIKE_UNLIKE_ERROR}}");
        }
     },
     
     /**
     * Report a wall content
     * @param {String} contentId the content id
     * @param {Function} [callback] the callback function to invoked after reporting content
     */
     report: function(contentId, callback)
     {
        var me = this;
        AmetysFront.Utils.confirm("{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_REPORT_TITLE}}", "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_REPORT_CONFIRM}}", function () {
            AmetysFront.ServerComm.callMethod({
	            role: 'org.ametys.plugins.workspaces.wall.WallContentManager',
	            methodName: 'reportContent',
	            parameters: [AmetysFront.getAppParameter('siteName'), contentId],
	            callback: {
	                handler: me._reportCb,
	                scope: me,
	                arguments: [callback]
	            }
	        });
        });
        
        return false;
     },
     
     _reportCb: function(success, args)
     {
        if (success)
        {
            AmetysFront.Utils.info("{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_REPORT_SUCCESS_TITLE}}", "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_REPORT_SUCCESS_MSG}}");
            var callback = args[0];
            if (typeof (callback) == 'function')
            {
                callback(data);
            }
        }
        else
        {
            AmetysFront.Utils.error("{{i18n plugin.workspaces:PLUGINS_WORKSPACES_WALL_CONTENT_REPORT_ERROR}}");
        }
     },
    
    _convertTextToRichText: function (text, openInNewWindow)
	{
	    text = text || "";
	    
	    text = text.replace(/&/g, "&amp;")
	        .replace(/</g, "&lt;")
	        .replace(/"/g, "&#034;")
	        .replace(/'/g, "&#039;")
	        .replace(
	            this._linkRe,
	            function(match, space, url) {
	              var hyperlink = url;
	              
                  var tc = "";
                  if (/(.*)[.,;:?!]$/.test(url)) // remove terminal character among .,;:?! from url
                  {
                     tc = url.substring(url.length-1);
                     url = url.substring(0, url.length-1);
                  }
                  
	              if (!hyperlink.match('^https?:\/\/')) {
	                hyperlink = 'http://' + hyperlink;
	              }
	              
	              return space + '<a ' + (openInNewWindow ? 'target="_blank" ' : '') + 'href="' + hyperlink + '">' + url + '</a>' + tc;
	            }
	          )
	          .replace(/\r?\n/g, "<br/>");
	          
	     return '<p>' + text + '</p>';
	},
    
    _convertRichTextToText: function (html)
	{
	    html = html || "";
	    return html.replace(/<br\/>/g, "BR").replace(/<br>/g, "\n")
	        .replace(/<[^>]*>/g, '');
	}
}
