<?xml version="1.0" encoding="UTF-8"?>
<!--
   Copyright 2021 Anyware Services

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
   -->
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                              xmlns:exslt="http://exslt.org/common"
                              xmlns:i18n="http://apache.org/cocoon/i18n/2.1"
                              xmlns:uri="org.ametys.core.util.URIUtils"
                              xmlns:ametys="org.ametys.web.transformation.xslt.AmetysXSLTHelper"
                              xmlns:resolver="org.ametys.cms.transformation.xslt.ResolveURIComponent"
                              xmlns:mail="org.ametys.plugins.workspaces.project.helper.MailXSLTHelper">
                              
    <xsl:import href="view://stylesheets/mail/abstract2notification.xsl"/>
    
    <xsl:variable name="maxItems">5</xsl:variable>
    <xsl:variable name="lowercase" select="'abcdefghijklmnopqrstuvwxyz.'" />
    <xsl:variable name="uppercase" select="'ABCDEFGHIJKLMNOPQRSTUVWXYZ_'" />
    
    <xsl:template name="title">
        <title>
            <i18n:translate>
                <i18n:text i18n:key="PROJECT_MAIL_NOTIFICATION_BODY_SUMMARY_TITLE_{@frequency}" i18n:catalogue="plugin.workspaces"/>
            </i18n:translate>
        </title>
    </xsl:template>
    
    <xsl:template name="body">
        <projects>
            <xsl:apply-templates select="project">
                <xsl:sort select="@title"/>
            </xsl:apply-templates>
        </projects>
    </xsl:template>
    
    <xsl:template match="project">
        <project name="{@name}">
            <title><xsl:value-of select="@title"/></title>
            <projectURL><xsl:value-of select="@url"/></projectURL>
            
            <sections>
                <!-- Force order manually -->
                <xsl:apply-templates select="events[@type = 'wallcontent.added']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'content.comment.validated']" mode="wallcontent">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'project.news.published']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'content.comment.validated']" mode="news">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'calendar.event.created']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'calendar.event.updated']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'calendar.event.deleting']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'resource.created']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'resource.commented']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'resource.updated']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'resource.renamed']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'task.created']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'task.assigned']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'task.closed.status.changed']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'task.deleting']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'task.commented']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'minisite.page.created']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'minisite.page.updated']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'minisite.page.renamed']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'minisite.page.deleted']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'member.added']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'thread.created']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'thread.post.created']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'forumthread.created']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'forumthread.modified']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'forumthread.closed']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'forumthread.deleted']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'forumthread.commented']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
                
                <xsl:apply-templates select="events[@type = 'forumthread.accepted']">
                    <xsl:with-param name="url" select="@url"/>
                </xsl:apply-templates>
            </sections>
        </project>
    </xsl:template>
    
    <xsl:template match="events">
    <xsl:param name="url"/>
        <section>
            <xsl:variable name="eventTypeId" select="translate(@type, $lowercase, $uppercase)"/>
            <xsl:variable name="nbEvents" select="count(event)"/>
            
            <xsl:apply-templates select="." mode="hint"/>
            
            <items>
                <xsl:apply-templates select="event[position() &lt;= $maxItems]">
                    <xsl:with-param name="url" select="$url"/>
                    <xsl:sort select="date" order="descending"/>
                </xsl:apply-templates>
                
                <xsl:variable name="moreItemsCount" select="$nbEvents - $maxItems"/>
                
                <xsl:if test="$moreItemsCount &gt; 0">
                    <more>
                        <xsl:variable name="multiple"><xsl:if test="$moreItemsCount &gt; 1">_MULTIPLE</xsl:if></xsl:variable>
                        <i18n:translate>
                            <i18n:text i18n:key="PROJECT_MAIL_NOTIFICATION_BODY_SUMMARY_{$eventTypeId}_SECTION_MORE{$multiple}" i18n:catalogue="plugin.workspaces"/>
                            <i18n:param name="nbOfItems"><xsl:value-of select="$moreItemsCount"/></i18n:param>
                        </i18n:translate>
                    </more>
                </xsl:if>
            </items>
        </section>
    </xsl:template>
    
    <!-- Override to handle event leading to multiple items -->
    <xsl:template match="events[@type = 'resource.created']">
        <section>
            <xsl:variable name="newFiles" select="event/files/files"/>
            <xsl:variable name="eventTypeId" select="translate(@type, $lowercase, $uppercase)"/>
            <xsl:variable name="nbEvents" select="count($newFiles)"/>
            
            <xsl:apply-templates select="." mode="hint">
                <xsl:with-param name="nbOfEvents" select="$nbEvents"/>
            </xsl:apply-templates>
            
            <items>
                <xsl:for-each select="$newFiles[position() &lt;= $maxItems]">
                    <xsl:sort select="../date" order="descending"/>
                    <item type="resource">
                        <title><xsl:value-of select="name"/></title>
                        <fileType>
                            <xsl:choose>
                                <xsl:when test="mimeType != ''"><xsl:value-of select="mail:getFileType(mimeType)"/></xsl:when>
                                <xsl:otherwise><xsl:value-of select="mail:getFileTypeFromId(id)"/></xsl:otherwise>
                            </xsl:choose>
                        </fileType>
                        <eventType><xsl:value-of select="translate(../../type , $lowercase , $uppercase)"/></eventType>
                        <lastModified><xsl:value-of select="../../date"/></lastModified>
                        <xsl:copy-of select="../../author"/>
                    </item>
                </xsl:for-each>
                
                <xsl:variable name="moreItemsCount" select="$nbEvents - $maxItems"/>
                
                <xsl:if test="$moreItemsCount &gt; 0">
                    <more>
                        <xsl:variable name="multiple"><xsl:if test="$moreItemsCount &gt; 1">_MULTIPLE</xsl:if></xsl:variable>
                        <i18n:translate>
                            <i18n:text i18n:key="PROJECT_MAIL_NOTIFICATION_BODY_SUMMARY_{$eventTypeId}_SECTION_MORE{$multiple}" i18n:catalogue="plugin.workspaces"/>
                            <i18n:param name="nbOfItems"><xsl:value-of select="$moreItemsCount"/></i18n:param>
                        </i18n:translate>
                    </more>
                </xsl:if>
            </items>
        </section>
    </xsl:template>
    
    <!-- Override to split the activity in separate section based on content type -->
    <xsl:template match="events[@type = 'content.comment.validated']" mode="wallcontent">
        <xsl:call-template name="comments-section">
            <xsl:with-param name="events" select="event[contentType = 'org.ametys.plugins.workspaces.Content.wallContent']"/>
            <xsl:with-param name="eventTypeId">WALLCONTENT_COMMENTED</xsl:with-param>
        </xsl:call-template>
    </xsl:template>
    
    <xsl:template match="events[@type = 'content.comment.validated']" mode="news">
        <xsl:call-template name="comments-section">
            <xsl:with-param name="events" select="event[contentType = 'org.ametys.plugins.workspaces.Content.projectNews']"/>
            <xsl:with-param name="eventTypeId">PROJECT_NEWS_COMMENTED</xsl:with-param>
        </xsl:call-template>
    </xsl:template>
    
    <xsl:template name="comments-section">
        <xsl:param name="events"/>
        <xsl:param name="eventTypeId"/>
        <xsl:if test="$events">
            <section>
                <xsl:variable name="nbOfEvents" select="count($events)"/>
                <xsl:variable name="multipleSuffix">
                    <xsl:if test="$nbOfEvents &gt; 1">_MULTIPLE</xsl:if>
                    <xsl:if test="$events[contentId/text() != $events[1]/contentId/text()]">_TARGETS</xsl:if>
                </xsl:variable>
                <hint>
                    <i18n:translate>
                        <i18n:text i18n:key="PROJECT_MAIL_NOTIFICATION_BODY_SUMMARY_{$eventTypeId}_SECTION_HINT{$multipleSuffix}" i18n:catalogue="plugin.workspaces"/>
                        <i18n:param name="nbOfEvents"><xsl:value-of select="$nbOfEvents"/></i18n:param>
                    </i18n:translate>
                </hint>
                
                <items>
                    <xsl:apply-templates select="$events[position() &lt;= $maxItems]">
                        <xsl:sort select="date" order="descending"/>
                    </xsl:apply-templates>
                
                    <xsl:variable name="moreItemsCount" select="$nbOfEvents - $maxItems"/>
                    
                    <xsl:if test="$moreItemsCount &gt; 0">
                        <more>
                            <xsl:variable name="multiple"><xsl:if test="$moreItemsCount &gt; 1">_MULTIPLE</xsl:if></xsl:variable>
                            <i18n:translate>
                                <i18n:text i18n:key="PROJECT_MAIL_NOTIFICATION_BODY_SUMMARY_{$eventTypeId}_SECTION_MORE{$multiple}" i18n:catalogue="plugin.workspaces"/>
                                <i18n:param name="nbOfItems"><xsl:value-of select="$moreItemsCount"/></i18n:param>
                            </i18n:translate>
                        </more>
                    </xsl:if>
                </items>
            </section>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="events" mode="hint">
        <xsl:param name="nbOfEvents" select="count(event)"/>
        <xsl:variable name="eventId" select="translate(@type, $lowercase, $uppercase)"/>
        <xsl:variable name="multipleSuffix"><xsl:if test="$nbOfEvents &gt; 1">_MULTIPLE</xsl:if></xsl:variable>
        <hint>
            <i18n:translate>
                <i18n:text i18n:key="PROJECT_MAIL_NOTIFICATION_BODY_SUMMARY_{$eventId}_SECTION_HINT{$multipleSuffix}" i18n:catalogue="plugin.workspaces"/>
                <i18n:param name="nbOfEvents"><xsl:value-of select="$nbOfEvents"/></i18n:param>
            </i18n:translate>
        </hint>
    </xsl:template>
    
    <xsl:template match="event[starts-with(type, 'resource')]">
    	<!-- a single resource event can target multiple files, hence multiple items -->
        <xsl:variable name="activity" select="."/>
        <xsl:for-each select="files/files|file">
            <item type="resource">
                <title><xsl:value-of select="name"/></title>
                <fileType>
                    <xsl:choose>
                        <xsl:when test="mimeType != ''"><xsl:value-of select="mail:getFileType(mimeType)"/></xsl:when>
                        <xsl:otherwise><xsl:value-of select="mail:getFileTypeFromId(id)"/></xsl:otherwise>
                    </xsl:choose>
                </fileType>
                <eventType><xsl:value-of select="translate($activity/type , $lowercase , $uppercase)"/></eventType>
                <xsl:if test="$activity/nbOfOccurrence &gt; 1">
                    <multiple>true</multiple>
                </xsl:if>
                <lastModified><xsl:value-of select="$activity/date"/></lastModified>
                <xsl:copy-of select="$activity/author"/>
            </item>
        </xsl:for-each>
    </xsl:template>
    
    <xsl:template match="event[type = 'resource.commented']">
        <xsl:if test="count(preceding-sibling::event[files/files/id = current()/files/files/id]) = 0">
            <item type="resource">
                <title><xsl:value-of select="files/files/name"/></title>
                <fileType>
                    <xsl:choose>
                        <xsl:when test="files/files/mimeType != ''"><xsl:value-of select="mail:getFileType(files/files/mimeType)"/></xsl:when>
                        <xsl:otherwise><xsl:value-of select="mail:getFileTypeFromId(files/files/id)"/></xsl:otherwise>
                    </xsl:choose>
                </fileType>
                <eventType><xsl:value-of select="translate(type , $lowercase , $uppercase)"/></eventType>
                <xsl:if test="nbOfOccurrence &gt; 1">
                    <multiple>true</multiple>
                </xsl:if>
                <lastModified><xsl:value-of select="lastModified"/></lastModified>
                <author>
                    <xsl:copy-of select="lastContributor/node()"/>
                </author>
                <comments>
                    <xsl:for-each select="../event[position() &lt;= $maxItems][files/files/id = current()/files/files/id]">
                        <comment>
                            <datetime><xsl:value-of select="date"/></datetime>
                            <author><xsl:copy-of select="author/node()"/></author>
                            <content><xsl:value-of select="commentSummary"/></content>
                        </comment>
                    </xsl:for-each>
                </comments>
            </item>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="events[@type = 'resource.commented']" mode="hint">
        <xsl:param name="nbOfEvents" select="count(event)"/>
        <xsl:variable name="eventId" select="translate(@type, $lowercase, $uppercase)"/>
        <xsl:variable name="multipleSuffix">
            <xsl:if test="$nbOfEvents &gt; 1">_MULTIPLE</xsl:if>
            <xsl:if test="event[files/files/id != current()/event[1]/files/files/id]">_TARGETS</xsl:if>
        </xsl:variable>
        <hint>
            <i18n:translate>
                <i18n:text i18n:key="PROJECT_MAIL_NOTIFICATION_BODY_SUMMARY_{$eventId}_SECTION_HINT{$multipleSuffix}" i18n:catalogue="plugin.workspaces"/>
                <i18n:param name="nbOfEvents"><xsl:value-of select="$nbOfEvents"/></i18n:param>
            </i18n:translate>
        </hint>
    </xsl:template>
    
    <xsl:template match="event[starts-with(type, 'calendar')]">
        <xsl:param name="url"/>
        <xsl:variable name="icslink" select="mail:getCalendarEventICSExportLink(eventId, $url)"/>
        <item type="calendar">
            <color><xsl:value-of select="mail:getCalendarColor(calendarId)"/></color>
            
            <startDate><xsl:value-of select="eventStartDate"/></startDate>
            <endDate><xsl:value-of select="eventEndDate"/></endDate>
            <fullDay><xsl:value-of select="eventFullDay"/></fullDay>
            
            <xsl:if test="eventRecurrence != 'NEVER' and eventRecurrence != ''">
                <recurrence>
                    <xsl:if test="eventRecurrence != 'NEVER' and eventRecurrence != ''">
                        <xsl:attribute name="until"><xsl:value-of select="eventRepeatUntil"/></xsl:attribute>
                    </xsl:if>
                    <xsl:value-of select="eventRecurrence"/>
                </recurrence>
            </xsl:if>
            <title><xsl:value-of select="eventTitle"/></title>
            <icsLink><xsl:value-of select="$icslink"/></icsLink>
        </item>
    </xsl:template>
    
    <xsl:template match="event[starts-with(type, 'member')]">
        <item type="member">
            <title><xsl:value-of select="member"/></title>
            <eventType><xsl:value-of select="translate(../@type , $lowercase , $uppercase)"/></eventType>
            <xsl:copy-of select="memberType"/>
            <xsl:copy-of select="author"/>
            <xsl:copy-of select="identity"/>
            <xsl:copy-of select="date"/>
        </item>
    </xsl:template>
    
    <xsl:template match="event[starts-with(type, 'task')]">
        <xsl:param name="url"/>
        <xsl:variable name="icslink" select="mail:getTaskICSExportLink(taskId, $url)"/>
        <xsl:variable name="hasDueDate" select="mail:hasDueDate(taskId)"/>
        <item type="task">
            <title><xsl:value-of select="taskTitle"/></title>
            <lastModified><xsl:value-of select="date"/></lastModified>
            <xsl:copy-of select="author"/>
            <eventType><xsl:value-of select="translate(../@type , $lowercase , $uppercase)"/></eventType>
            <xsl:if test="nbOfOccurrence &gt; 1">
                <multiple>true</multiple>
            </xsl:if>
            <xsl:if test="$hasDueDate != ''">
	            <icsLink><xsl:value-of select="$icslink"/></icsLink>
            </xsl:if>
        </item>
    </xsl:template>
    
    <xsl:template match="event[type = 'task.commented']">
    
	    <xsl:param name="url"/>
	    <xsl:if test="count(preceding-sibling::event[taskId = current()/taskId]) = 0">
	        <xsl:variable name="icslink" select="mail:getTaskICSExportLink(taskId, $url)"/>
	        <xsl:variable name="hasDueDate" select="mail:hasDueDate(taskId)"/>
	        <xsl:variable name="taskId" select="taskId"/>
	        <item type="task">
	            <title><xsl:value-of select="taskTitle"/></title>
                <lastModified><xsl:value-of select="creationDate"/></lastModified>
                <author>
                    <xsl:copy-of select="taskAuthor/node()"/>
                </author>
	            <eventType><xsl:value-of select="translate(../@type , $lowercase , $uppercase)"/></eventType>
	            <xsl:if test="nbOfOccurrence &gt; 1">
	                <multiple>true</multiple>
	            </xsl:if>
	            <xsl:if test="$hasDueDate != ''">
	                <icsLink><xsl:value-of select="$icslink"/></icsLink>
	            </xsl:if>
	            <xsl:variable name="allCommentActivies" select="../event[position() &lt;= $maxItems]"/>
	            
	            <comments>
	                <xsl:for-each select="$allCommentActivies[taskId = $taskId]">
	                    <comment>
	                        <datetime><xsl:value-of select="date"/></datetime>
	                        <author><xsl:copy-of select="author/node()"/></author>
	                        <content><xsl:value-of select="commentSummary"/></content>
	                    </comment>
	                </xsl:for-each>
	            </comments>
	        </item>
	    </xsl:if>
    </xsl:template>
    
    <xsl:template match="events[@type = 'task.commented']" mode="hint">
        <xsl:param name="nbOfEvents" select="count(event)"/>
        <xsl:variable name="eventId" select="translate(@type, $lowercase, $uppercase)"/>
        <xsl:variable name="multipleSuffix">
            <xsl:if test="$nbOfEvents &gt; 1">_MULTIPLE</xsl:if>
            <xsl:if test="event[taskId != current()/event[1]/taskId]">_TARGETS</xsl:if>
        </xsl:variable>
        <hint>
            <i18n:translate>
                <i18n:text i18n:key="PROJECT_MAIL_NOTIFICATION_BODY_SUMMARY_{$eventId}_SECTION_HINT{$multipleSuffix}" i18n:catalogue="plugin.workspaces"/>
                <i18n:param name="nbOfEvents"><xsl:value-of select="$nbOfEvents"/></i18n:param>
            </i18n:translate>
        </hint>
    </xsl:template>
    
    <xsl:template match="event[starts-with(type, 'wallcontent')]">
        <xsl:if test="contentSummary != ''">
            <item type="content">
                <title><xsl:value-of select="author/fullname"/></title>
                <datetime><xsl:value-of select="date"/></datetime>
                <xsl:copy-of select="author"/>
                <content><xsl:value-of select="contentSummary"/></content>
                <xsl:if test="ametys:hasValue(contentId, 'illustration/image')">
                    <xsl:variable name="img" select="ametys:contentAttribute(contentId, 'illustration/image')"/>
                    <images>
                        <image>
                            <src><xsl:value-of select="resolver:resolveBoundedImageAsBase64($img/@type, $img/@uri, 340, 490)"/></src>
                        </image>
                    </images>
                </xsl:if>
            </item>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="event[type = 'project.news.published']">
        <item type="content">
            <title><xsl:value-of select="contentTitle"/></title>
            <date><xsl:value-of select="date"/></date>
            <xsl:copy-of select="author"/>
            <content><xsl:value-of select="contentSummary"/></content>
            <xsl:if test="contentTags/*">
                <tags>
                    <xsl:for-each select="contentTags/*">
                        <tag><xsl:value-of select="."/></tag>
                    </xsl:for-each>
                </tags>
            </xsl:if>
            <xsl:if test="ametys:hasValue(contentId, 'illustration/image')">
                <xsl:variable name="img" select="ametys:contentAttribute(contentId, 'illustration/image')"/>
                <images>
                    <image>
                        <src><xsl:value-of select="resolver:resolveBoundedImageAsBase64($img/@type, $img/@uri, 340, 490)"/></src>
                    </image>
                </images>
            </xsl:if>
        </item>
    </xsl:template>
    
    <xsl:template match="event[starts-with(type, 'minisite.')]">
        <item type="page">
            <title><xsl:value-of select="pageTitle"/></title>
            <id><xsl:value-of select="pageId"/></id>
            <date><xsl:value-of select="date"/></date>
            <xsl:copy-of select="author"/>
            <eventType><xsl:value-of select="translate(../@type , $lowercase , $uppercase)"/></eventType>
            <xsl:if test="nbOfOccurrence &gt; 1">
                <multiple>true</multiple>
            </xsl:if>
        </item>
    </xsl:template>
    
    <xsl:template match="event[../@type = 'content.comment.validated']" priority="5">
        <!-- We check that it's the first comment activity of this content -->
        <xsl:variable name="contentId" select="contentId"/>
        <xsl:variable name="cType" select="contentType"/>
        
        <xsl:if test="not(preceding-sibling::event[contentId = $contentId])">
            <!-- If so, we start by rendering the content -->
            <item type="content">
                <title>
                    <xsl:choose>
                        <xsl:when test="contentType = 'org.ametys.plugins.workspaces.Content.projectNews'">
                            <xsl:value-of select="contentTitle"/>
                        </xsl:when>
                        <xsl:when test="contentType = 'org.ametys.plugins.workspaces.Content.wallContent'">
                            <xsl:value-of select="contentCreator/fullname"/>
                        </xsl:when>
                    </xsl:choose>
                </title>
                <contentId><xsl:value-of select="$contentId"/></contentId>
                <contentType><xsl:value-of select="$cType"/></contentType>
                <eventType><xsl:value-of select="translate(../@type , $lowercase , $uppercase)"/></eventType>
                <date>
                    <xsl:value-of select="contentCreationDate"/>
                </date>
                <author>
                    <xsl:copy-of select="contentCreator/*"/>
                </author>
                <content><xsl:value-of select="contentSummary"/></content>
                
                <xsl:if test="contentTags/*">
                    <tags>
                        <xsl:for-each select="contentTags/*">
                            <tag><xsl:value-of select="."/></tag>
                        </xsl:for-each>
                    </tags>
                </xsl:if>
                <xsl:if test="ametys:hasValue($contentId, 'illustration/image')">
                    <xsl:variable name="img" select="ametys:contentAttribute($contentId, 'illustration/image')"/>
                    <images>
                        <image>
                            <src><xsl:value-of select="resolver:resolveBoundedImageAsBase64($img/@type, $img/@uri, 340, 490)"/></src>
                        </image>
                    </images>
                </xsl:if>
                <!-- Then we retrieve the $maxItems first comment activities that are linked to this content -->
                <comments>
                    <!-- First get maxItems comment activities among all comments activity of a same content type -->
                    <xsl:variable name="allCommentActivies" select="../event[contentType = $cType][position() &lt;= $maxItems]"/>
                    
                    <!-- Iterate over maxItems comment activities dealing with this content -->
                    <xsl:for-each select="$allCommentActivies[contentId = $contentId]">
                         <xsl:choose>
                            <!-- 1: Case of a sub-comment whose parent has not already been processed (either as parent or root comment) -->
                            <xsl:when test="parentComment and not(preceding-sibling::event[contentId = $contentId and parentComment/commentId = current()/parentComment/commentId]) and not(preceding-sibling::event[contentId = $contentId and commentId = current()/parentComment/commentId])">
                                <comment>
                                    <xsl:variable name="currentParentCommentId" select="parentComment/commentId"/>
                                    <!-- Parent comment is an old comment if it is not part of comment activities as a root comment -->
                                    <xsl:if test="not(../event[contentId = $contentId and commentId = $currentParentCommentId])">
                                        <xsl:attribute name="new">false</xsl:attribute>
                                    </xsl:if>
                                    <id><xsl:value-of select="$currentParentCommentId"/></id>
                                    <datetime><xsl:value-of select="parentComment/date"/></datetime>
                                    <xsl:copy-of select="parentComment/author"/>
                                    <content><xsl:value-of select="parentComment/commentSummary"/></content>
                                    <test><xsl:value-of select="commentSummary"/></test>
                                    <sub-comments>
                                        <!-- Group all sub-comments linked to the same parent comment (among  $maxItems activities) -->
                                        <xsl:for-each select="$allCommentActivies[contentId = $contentId and parentComment/commentId = $currentParentCommentId]">
                                            <comment>
                                                <id><xsl:value-of select="commentId"/></id>
                                                <datetime><xsl:value-of select="date"/></datetime>
                                                <xsl:copy-of select="author"/>
                                                <content><xsl:value-of select="commentSummary"/></content>
                                            </comment>
                                        </xsl:for-each>
                                    </sub-comments>
                                </comment>
                            </xsl:when>
                            <!-- 2: Case of a root comment: checks that it has not already been processed in the case of a sub-comment -->
                            <xsl:when test="not(parentComment) and not(preceding-sibling::event[contentId = $contentId and parentComment/commentId = current()/commentId])">
                                <xsl:variable name="currentCommentId" select="commentId"/>
                                <comment>
                                    <id><xsl:value-of select="$currentCommentId"/></id>
                                    <datetime><xsl:value-of select="date"/></datetime>
                                    <xsl:copy-of select="author"/>
                                    <content><xsl:value-of select="commentSummary"/></content>
                                    <xsl:if test="$allCommentActivies[contentId = $contentId and parentComment/commentId = $currentCommentId]"> 
                                        <sub-comments>
                                            <!-- Group all sub-comments linked to this comment (among  $maxItems activities) -->
                                            <xsl:for-each select="$allCommentActivies[contentId = $contentId and parentComment/commentId = $currentCommentId]">
                                                <comment>
                                                    <id><xsl:value-of select="commentId"/></id>
                                                    <datetime><xsl:value-of select="date"/></datetime>
                                                    <xsl:copy-of select="author"/>
                                                    <content><xsl:value-of select="commentSummary"/></content>
                                                </comment>
                                            </xsl:for-each>
                                        </sub-comments>
                                    </xsl:if>
                                </comment>
                            </xsl:when>
                            <xsl:otherwise>
                                <!-- Nothing to do, it has been already processed in case 1 or 2-->
                            </xsl:otherwise>
                         </xsl:choose>
                    </xsl:for-each>
                </comments>
            </item>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="event[starts-with(type, 'forumthread')]">
        <xsl:variable name="threadId" select="threadId"/>
        
        <xsl:if test="not(preceding-sibling::event[threadId = $threadId])">
            <item type="forum">
                <title><xsl:value-of select="threadTitle"/></title>
                <xsl:copy-of select="date"/>
                <xsl:copy-of select="author"/>
                <xsl:copy-of select="threadCategory"/>
                <eventType><xsl:value-of select="translate(../@type , $lowercase , $uppercase)"/></eventType>
                
                <xsl:if test="type = 'forumthread.commented' or type ='forumthread.accepted'">
                    <xsl:variable name="allCommentActivies" select="../event[position() &lt;= $maxItems]"/>
                    
                    <comments>
                        <xsl:for-each select="$allCommentActivies[threadId = $threadId]">
                            <comment>
                                <id><xsl:value-of select="commentId"/></id>
                                <datetime><xsl:value-of select="commentCreationDate"/></datetime>
                                <author><xsl:copy-of select="commentAuthor/node()"/></author>
                                <xsl:copy-of select="content"/>
                                <xsl:if test="type = 'forumthread.accepted'">
                                    <accepted>true</accepted>
                                </xsl:if>
                            </comment>
                        </xsl:for-each>
                    </comments>
                </xsl:if>
            </item>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="event"/>
      
</xsl:stylesheet>
