/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import i18n from 'i18n';

// A prefix to have before full name of a mentioned user
export const MENTIONED_USER_PREFIX = '@';

/**
 * export plain text to syntax text by replacing mention names with their ids
 * @param {string} text the plain text
 * @param {Array} mentionList the list of mentions with their names and ids
 * @returns {string} the syntax text
 * @example
 * "Hello @John Doe" with mentionList [{name: "@John Doe", id: "@(jdoe#users)"}] will return "Hello @(jdoe#users)"
 */
export function textToSyntaxText(text, mentionList)
{
    var syntaxText = text;
    for (let i = 0; i < mentionList.length; i++)
    {
        const mention = mentionList[i];
        syntaxText = syntaxText.replaceAll(mention.name, mention.id);
    }
    return syntaxText;
}

/**
 * export syntax text to formatted text by replacing mention ids with their names wrapped in a span
 * @param {string} syntaxText the syntax text
 * @param {Array} mentionList the list of mentions with their names and ids
 * @returns {string} the formatted text
 * @example
 * "Hello @(jdoe#users)" with mentionList [{name: "John Doe, id: "@(jdoe#users)", mentionCategory: "ametys-mention-allowed"}]
 * will return 'Hello <span class="mention ametys-mention-allowed">John Doe</span>'
 */
export function syntaxToFormattedText(syntaxText, mentionList)
{
    var text = syntaxText.replace(/\n/g, '<br/>');
    for (let i = 0; i < mentionList.length; i++)
    {
        const mention = mentionList[i];
        let mentionTooltip = mention.mentionCategory == 'ametys-mention-not-allowed' ? 'data-ametys-tooltip="' + i18n.PLUGINS_WORKSPACES_PROJECTS_MENTION_USER_NOT_MEMBER_TOOLTIP + '"': '';
        text = text.replaceAll(mention.id, `<span ${mentionTooltip} class="mention ${mention.mentionCategory}">${mention.name}</span>`);
    }
    return text;
}

/**
 * export syntax text to editable text by replacing mention ids with their names
 * @param {string} syntaxText the syntax text
 * @param {Array} mentionList the list of mentions with their names and ids
 * @returns {string} the editable text
 * @example
 * "Hello @(jdoe#users)" with mentionList [{name: "John Doe, id: "@(jdoe#users)", mentionCategory: "ametys-mention-allowed"}]
 * will return 'Hello John Doe'
 */
export function syntaxToEditText(syntaxText, mentionList)
{
    var text = syntaxText;
    for (let i = 0; i < mentionList.length; i++)
    {
        const mention = mentionList[i];
        text = text.replaceAll(mention.id, mention.name);
    }
    return text;
}

/**
 * Compute the filtered users based on the search text.
 * The search is case insensitive and matches both the user name and id.
 * @param {string} userSearchText the search text
 * @param {Array} members the list of members to filter
 * @returns {Array} the filtered list of members
 * @example
 * computeFilteredUsers("john", [{name: "John Doe", id: "jdoe#users"}, {name: "Jane Smith", id: "jsmith#users"}]) will return [{name: "John Doe", id: "jdoe#users"}]
 */
export function computeFilteredUsers(userSearchText, members)
{
    if (!userSearchText) return [];

    return members.filter(user =>
        user.name.toLowerCase().includes(userSearchText.toLowerCase()) ||
        (user.login && user.login.toLowerCase().includes(userSearchText.toLowerCase()))
    )
    .sort((a, b) => a.sortablename.localeCompare(b.sortablename));
}

/**
 * Get the mention match from the text.
 * A mention match is defined as a string that starts with "@" followed by at least 2 word characters
 * @param {string} text the text to search for a mention
 * @returns {string} the mention match without the "@" symbol, or null if no match is found
 * @example
 * getMentionMatch("Hello @John") will return "john"
 */
export function getMentionMatch(text)
{
    const parts = text.split(MENTIONED_USER_PREFIX);
    if (parts.length < 2) return null;
    const last = parts[parts.length - 1];
    return last.length >= 2 ? last.toLowerCase() : null;
}

/**
 * Transform text from syntax to render view
 * @param {string} text the text to transform
 */
export function userListToMentionList(userList, members, moduleName)
{
    return userList.map(user => {

        // get the member to check if he has read access to the module
        let member = members.find(member => member.id === (user.login + "#" + user.populationId));
        return {
            name: MENTIONED_USER_PREFIX + user.fullname,
            id: "@(" + user.login + "#" + user.populationId + ")",
            mentionCategory: mentionClass(member, moduleName)
        };
    });
}

/**
 * Get the CSS class for a mention based on whether the member is the current user or not
 * used to display correct clor for mentions
 */
export function mentionClass(member, moduleName)
{
    if (member && member.isCurrentUser)
    {
      return "ametys-mention-own";
    }
    else if (member && member.readAccess[moduleName])
    {
      return "ametys-mention-allowed";
    }
    else
    {
      return "ametys-mention-not-allowed";
    }
}

/**
 * Transform the selected user text in the comment by replacing the search text with the selected user name
 * @param {Object} user the selected user
 * @param {string} comment the current comment text
 * @param {string} textAreaId the id of the textarea to update
 * @param {string} userSearchText the search text used to filter users
 * @returns {string} the updated comment text
 */
export function transformSelectedUserText(user, comment, textAreaId, userSearchText)
{

  const beforeCaret = getTextBeforeCaret(textAreaId);

  const atIndex = beforeCaret.lastIndexOf(MENTIONED_USER_PREFIX);
  if (atIndex !== -1)
  {
    const beforeAt = comment.substring(0, atIndex);
    const afterSearchText = comment.substring(atIndex + userSearchText.length + 1);


    const textarea = document.getElementById(textAreaId);

    // set the caret position after the inserted user name
    setTimeout(() => {

      // +1 for the @ and +1 for the space after the user name
      textarea.selectionStart = beforeAt.length + 2 + user.name.length;
      textarea.selectionEnd = beforeAt.length + 2 + user.name.length;
    }, 1);

    return `${beforeAt}${MENTIONED_USER_PREFIX}${user.name} ${afterSearchText}`;
  }
  else
  {
    return comment
  }
}

/**
 * Get the coordinates of the char used to mention someone in a TextAea, so we can display a list of user around it
 * @param {string} textAreaId the id of the textarea to search text in
 * @param {string} parentId the id of the div wraping the textarea, used to compute offset
 */
export function getMentionCharCoordinatesForTextArea(textAreaId, parentId)
{
  // Returns [y, x] of last '@' before caret in textarea, both negative for v-menu offset
  const textarea = document.getElementById(textAreaId);
  const parent = document.getElementById(parentId);

  if (!textarea || !parent) return [0, 0];
  const caretPos = textarea.selectionStart;
  const textBeforeMentionChar = textarea.value.substring(0, caretPos);
  const lastMentionChar = textBeforeMentionChar.lastIndexOf(MENTIONED_USER_PREFIX, caretPos - 1);
  // get value between last @ and next space or end of text, as if there is a text just after caret it should be taken into account
  const textAftetLastMentionChar =  textarea.value.substring(lastMentionChar);
  const textBetweenLastMentionCharAndNextSpace = textAftetLastMentionChar.split(' ')[0];


  if (lastMentionChar === -1) return [0, 0];
  // Mirror div technique
  const style = window.getComputedStyle(textarea);
  const div = document.createElement('div');
  div.style.position = 'absolute';
  div.style.visibility = 'hidden';
  div.style.whiteSpace = 'pre-wrap';
  div.style.wordWrap = 'break-word';
  div.style.font = style.font;
  div.style.padding = style.padding;
  div.style.border = style.border;
  div.style.width = textarea.offsetWidth + 'px';
  div.style.lineHeight = style.lineHeight;
  div.textContent = textBeforeMentionChar.substring(0, lastMentionChar);
  document.body.appendChild(div);
  const span = document.createElement('span');
  span.textContent = textBetweenLastMentionCharAndNextSpace;
  div.appendChild(span);
  const rect = span.getBoundingClientRect();
  const textareaRect = textarea.getBoundingClientRect();

  const parentRect = parent.getBoundingClientRect();
  let offsetX = textareaRect.left - parentRect.left;

  const x = rect.left + offsetX;
  const y = rect.top - textareaRect.top;
  const divrect = div.getBoundingClientRect();
  document.body.removeChild(div);
  return {
    x: Math.round(x),
    y: Math.round(rect.top - divrect.top)
  }
}


/**
 * Get the coordinates of the char used to mention someone in TinyMCE editor, so we can display a list of user around it
 * @param {Object} editor TinyMCE editor
 */
export function getMentionCharCoordinatesForTinyMCE(editor)
{
    if (!editor || !editor.selection) return;
    const rng = editor.selection.getRng();
    const text = rng.startContainer.textContent;
    if (!text) return null;

    // Find last '@' before caret
    const cursorPos = rng.startOffset;
    const beforeCursor = text.substring(0, cursorPos);
    const atIndex = beforeCursor.lastIndexOf(MENTIONED_USER_PREFIX);
    if (atIndex === -1) return null;

    // create a range with this '@'
    const newRng = document.createRange();
    newRng.setStart(rng.startContainer, atIndex);
    newRng.setEnd(rng.startContainer, atIndex + 1);

    const rects = newRng.getClientRects();
    if (rects.length > 0) {
      return rects[0];
    }
    return null;
}

/**
 * Get the value of a textArea before selection
 * @param {String} textAreaId the textarea id
 */
export function getTextBeforeCaret(textAreaId)
{
  const textarea = document.getElementById(textAreaId);
  const caretPos = textarea.selectionStart;
  return textarea.value.substring(0, caretPos);
}
