/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront'
import i18n from 'i18n'
import { callMethod } from '@common/helper/ServerCommHelper'
import { useFoldersStore } from '@/stores/folders'
import { useFilesStore } from '@/stores/files'
import { useSelectedItemsStore } from '@/stores/selectedItems'

/**
 * Pinia store for moving folders and files
 */
export const useMoveFoldersAndFilesStore = defineStore('moveFoldersAndFiles', {
    state: () => ({
        folders: [], // List of folders in the current folder
        currentFolder: {}, // The current folder object
        treeOpenedFolders: [], // List of opened folders in the tree
        treeActiveFolder: [] // List of active folders in the tree
    }),
    actions: {

        /**
         * Create a new folder in the current folder
         * @param {Object} folder the folder object to create
         */
        createFolder(folder) {
          if (this.currentFolder.id == folder.parentId)
          {
              // In the case were a new folder is created in newly created folder,
              // currentFolder is set with the parent id but children is already added, check if the folder is not already in the list
              if (!this.currentFolder.children || !this.currentFolder.children.some(child => child.id === folder.id)) {

                  if (this.currentFolder.hasOwnProperty('children')) {
                    this.currentFolder.children.push(folder)
                  } else {
                    Vue.set(this.currentFolder, 'children', [folder])
                  }

                  this.folders = this.currentFolder.children

                  AmetysFront.Event.fire('rerender-file-tree');
              }
          }
        },

        /**
         * Expand folder in tree
         * @param {Object} folder the folder object
         */
        async expandFolder(folder)
        {
            try
            {
                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFoldersAndFiles',
                    parameters: [folder.id]
                })
                folder.children = data.children
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Open folder from breadcrumb
         * @param {Object} folder the folder object
         */
        async breadOpenFolder(folder)
        {
            try
            {
                await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFoldersAndFiles',
                    parameters: [folder.id]
                })

                this.currentFolder = folder
                this.folders = folder.children ? folder.children : []
                this.treeOpenedFolders.push(folder)
                this.treeActiveFolder = [folder]
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Open folder
         * @param {Object} folder the folder object
         */
        async openFolder(folder)
        {
            try
            {
                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'getFoldersAndFiles',
                    parameters: [folder.id]
                })

                if (data.hasOwnProperty('children'))
                {
                    let children = data.children
                    if (folder.children)
                    {
                        for (let child of children)
                        {
                            let f = folder.children.find(f => f.id == child.id)
                            if (f)
                            {
                                child.children = f.children
                            }
                        }
                    }
                    folder.children = children
                }
                else
                {
                    if (folder.hasOwnProperty('children'))
                    {
                        delete folder.children
                    }
                }

                this.currentFolder = folder
                this.folders = folder.children ? folder.children : []
                this.treeOpenedFolders.push(folder)
                this.treeActiveFolder = [folder]

                AmetysFront.Event.fire('reset-filter')
                AmetysFront.Event.fire('hideMobileHeader')
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Copy folders and files
         * @param {Object} payload the payload containing selected folders and files ids, and the target folder id
         * @param {Array} payload.selectedFoldersIds the ids of the selected folders
         * @param {Array} payload.selectedFilesIds the ids of the selected files
         * @param {string} payload.toFolderId the id of the target folder
         * @returns {boolean}
         */
        async copyFoldersAndFiles(payload)
        {
            const foldersStore = useFoldersStore();
            const filesStore = useFilesStore();
            const selectedItemsStore = useSelectedItemsStore();
            let success = false

            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_STARTED
            })

            try
            {
                let objectIds = [...payload.selectedFilesIds]
                for (let folderId of payload.selectedFoldersIds)
                {
                    objectIds.push(folderId)
                }

                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'copyFoldersAndFiles',
                    parameters: [objectIds, payload.toFolderId]
                })

                if (data.message == 'locked')
                {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_ERROR_LOCKED_MESSAGE,
                        details: null
                    })
                }
                else if (data.message == "already-exist")
                {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_ERROR_ALREADY_EXIST_MESSAGE + " " + data["uncopied-objects"],
                        details: null
                    })
                }

                let copiedObjects = data["copied-objects"]
                if (copiedObjects && copiedObjects.length > 0)
                {
                    success = true
                    let copiedFolders = data["copied-folders"]
                    if (copiedFolders && copiedFolders.length > 0)
                    {
                        foldersStore.refreshFolderByIdAfterCopy(payload.toFolderId)
                    }
                    selectedItemsStore.unselectAll();
                    AmetysFront.Event.fire('loaderEnd', {
                      text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_ENDED
                    });

                    return success;
                }
                else if (!data.message)
                {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_ERROR_MESSAGE,
                        details: null
                    })
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_ERROR_TITLE,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_COPY_ERROR_MESSAGE,
                    details: null
                })
            }

            return success
        },

        /**
         * Move folders and files
         * @param {Object} payload the payload containing selected folders and files ids, and the target folder id
         * @param {Array} payload.selectedFoldersIds the ids of the selected folders
         * @param {Array} payload.selectedFilesIds the ids of the selected files
         * @param {string} payload.toFolderId the id of the target folder
         * @returns {boolean}
         */
        async moveFoldersAndFiles(payload)
        {
            const foldersStore = useFoldersStore();
            const filesStore = useFilesStore();
            const selectedItemsStore = useSelectedItemsStore();
            let success = false

            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_STARTED
            })

            try
            {
                let objectIds = [...payload.selectedFilesIds]
                for (let folderId of payload.selectedFoldersIds)
                {
                    objectIds.push(folderId)
                }

                let data = await callMethod({
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'moveObjects',
                    parameters: [objectIds, payload.toFolderId]
                })

                if (data.message == 'locked')
                {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_LOCKED_MESSAGE,
                        details: null
                    })
                }
                else if (data.message == "already-exist")
                {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_ALREADY_EXIST_MESSAGE + " " + data["unmoved-objects"],
                        details: null
                    })
                }

                let movedObjects = data["moved-objects"]
                if (movedObjects && movedObjects.length > 0)
                {
                    success = true

                    foldersStore.updateFoldersAfterMove(data["moved-folders"], payload.toFolderId);
                    filesStore.updateFilesAfterMove(data["moved-files"]);
                    selectedItemsStore.unselectAll();
                    AmetysFront.Event.fire('loaderEnd', {
                      text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ENDED
                    });

                    return success;

                }
                else if (!data.message)
                {
                    AmetysFront.Event.fire('loaderFail', {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_MESSAGE,
                        details: null
                    })
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_TITLE,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_MESSAGE,
                    details: null
                })
            }

            return success
        },

        /**
         * Show current folders structure
         */
        showCurrentFoldersStructure()
        {
            const foldersStore = useFoldersStore()
            this.currentFolder = foldersStore.currentFolder
            this.treeOpenedFolders = JSON.parse(JSON.stringify(foldersStore.treeOpenedFolders))
            this.folders = foldersStore.folders
            this.treeActiveFolder = foldersStore.treeActiveFolder
        },

        setTreeOpenedFolders(treeOpenedFolders) {
            this.treeOpenedFolders = treeOpenedFolders
        }
    }
})
