/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia';
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import { transformToMember } from '@common/helper/MemberHelper.js';
import i18n from 'i18n';
import { useSearchUsersStore } from '@/stores/searchUsers'


const projectName = AmetysFront.getAppParameter('projectName');


/*
 * Sort the members by manager, group and name : first the managers, then the groups, then the users
 * sort alphabetically by name for the same category
 * @param {Object[]} members The members to sort
 */
function _sort(members)
{
    members.sort(function(c1,c2) {
        if (c1.isManager && !c2.isManager) return -1;
        if (!c1.isManager && c2.isManager) return 1;

        if (c1.isGroup && !c2.isGroup) return -1;
        if (!c1.isGroup && c2.isGroup) return 1;

        if (c1.name == c2.name) return 0;
        else if (c1.name < c2.name) return -1;
        else return 1;
    });
}

/**
 * Get the text error for existing users
 * @param {Object[]} existingUsers The existing users
 */
function _getExistingUserText(existingUsers)
{
    let text = "";
    let length = existingUsers.length;
    if (length == 1)
    {
        text = i18n.PLUGINS_WORKSPACES_MEMBERS_ADD_MEMBER_ALREADY_EXIST_MSG_ONE + " " + existingUsers[0].fullname + " (" + existingUsers[0].email + ")"
    }
    else if (length > 1)
    {
        text = i18n.PLUGINS_WORKSPACES_MEMBERS_ADD_MEMBER_ALREADY_EXIST_MSG_SEVERAL + " ";
        for (let i in existingUsers)
        {
            text += existingUsers[i].fullname + " (" + existingUsers[i].email + ")";
            if (i != length - 1)
            {
                text += ", ";
            }
        }
    }
    return text;
}

/**
 * Pinia store for handling project members
 */
export const useMembersStore = defineStore('members',{
    state: () => ({
        rights: {}, // rights of the current user
        members: [], // members of the current project
        profiles: [], // list of available profiles to set
        modules: [] // list of modules of the current project
    }),
    actions: {

        /**
         * Get the user rights
         * @return The user rights
         */
        async getUserRights()
        {
            if (this.rights === null || this.rights.view === undefined)
            {
                try
                {
                    
                    let result = await callMethod({
                        role: 'org.ametys.plugins.workspaces.members.ProjectMemberManager',
                        methodName: 'getMemberModuleRights',
                        parameters: [ projectName ]
                    });
                    if (result.success)
                    {
                        this.rights = {
                            'view': result.rights.view || false,
                            'add': result.rights.add || false,
                            'edit': result.rights.edit || false,
                            'delete': result.rights.delete || false
                        };
                    }
                    else
                    {
                        if (result["message"] == "unknown-project")
                        {
                            AmetysFront.Event.fire('loaderFail', { 
                                title: i18n.PLUGINS_WORKSPACES_MEMBERS_GET_RIGHTS_ERROR,
                                text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_PROJECT_ERROR_TEXT,
                                details: null
                            });
                        }
                    }
                }
                catch (e)
                {
                    AmetysFront.Event.fire('loaderFail', { 
                        title: i18n.PLUGINS_WORKSPACES_MEMBERS_GET_RIGHTS_ERROR,
                        text : i18n.PLUGINS_WORKSPACES_MEMBERS_GENERAL_ERROR_TEXT,
                        details: e
                    });
                }
            }

            return this.rights;
        },

        /**
         * Load the project members and store them
         */
        async loadProjectMembers() {
            try
            {
                const rights = await this.getUserRights();
                if (!rights.view)
                {
                    return;
                }
                else
                {
                    const result = await callMethod({
                        role: 'org.ametys.plugins.workspaces.members.ProjectMemberManager',
                        methodName: 'getProjectMembers',
                        parameters: [ projectName, AmetysFront.getAppParameter('sitemapLanguage')]
                    });
                    if (result.success)
                    {
                        this.members = result.members.map(transformToMember);
                        _sort(this.members);
                    }
                    else
                    {
                        if (result["message"] == "unknown-project")
                        {
                            AmetysFront.Event.fire('loaderFail', { 
                                title: i18n.PLUGINS_WORKSPACES_MEMBERS_GET_MEMBERS_ERROR,
                                text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_PROJECT_ERROR_TEXT,
                                details: null
                            });
                        }
                    }
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_MEMBERS_GET_MEMBERS_ERROR,
                    text : i18n.PLUGINS_WORKSPACES_MEMBERS_GENERAL_ERROR_TEXT,
                    details: e
                });
            }
        },

        /**
         * Add existing members or invite future ones by email
         * @param {Object[]} members The members to add (users or groups)
         * @param {Object[]} emails The invitation emails
         */
        async addMembers(members, emails) {
           const newMembers = [];
           members.forEach(member => {
              newMembers.push({
                id: member.id,
                type: member.isGroup ? 'group' : 'user'
              })
           });

           try {
               AmetysFront.Event.fire('loaderStart', {
                   text: i18n.PLUGINS_WORKSPACES_MEMBERS_ADD_MEMBERS_START
               });

               let result = await callMethod({
                     role: 'org.ametys.plugins.workspaces.members.ProjectMemberManager',
                     methodName: 'addMembers',
                     parameters: [
                         projectName,
                         newMembers,
                         emails
                     ]
                })
                if (result.success)
                {
                    if (result['members-added'])
                    {
                        if (this.members === undefined)
                        {
                            this.members = [];
                        }
                        
                        result['members-added'].map(transformToMember).forEach(member => {
                            this.members.push(member);
                        });
                        _sort(this.members);
                    }

                    AmetysFront.Event.fire('loaderEnd', { 
                        text : i18n.PLUGINS_WORKSPACES_MEMBERS_ADD_MEMBERS_SUCCESS
                    });

                    if (result['existing-users'] && result['existing-users'].length > 0)
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: result['existing-users'].length > 1 ? i18n.PLUGINS_WORKSPACES_MEMBERS_ADD_MEMBER_ALREADY_EXIST_SEVERAL_ERROR : i18n.PLUGINS_WORKSPACES_MEMBERS_ADD_MEMBER_ALREADY_EXIST_ERROR,
                            text : _getExistingUserText(result['existing-users']),
                            details: null
                        });
                    }
                }
                else
                {
                    if (result['invite-error'])
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: i18n.PLUGINS_WORKSPACES_MEMBERS_ADD_MEMBERS_ERRORINVITE,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_GENERAL_ERROR_TEXT,
                            details: null
                        });
                    }

                    if (result['existing-users'] && result['existing-users'].length > 0)
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: i18n.PLUGINS_WORKSPACES_MEMBERS_ADD_MEMBER_ALREADY_EXIST_ERROR,
                            text : _getExistingUserText(result['existing-users']),
                            details: null
                        });
                    }

                    if (result['unknown-project'] || result['unknown-users'].length > 0 || result['unknown-groups'].length > 0)
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: i18n.PLUGINS_WORKSPACES_MEMBERS_ADD_MEMBERS_ERROR,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_GENERAL_ERROR_TEXT,
                            details: null
                        });
                    }
                }
           }
           catch (e)
           {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_MEMBERS_ADDMEMBERS_ERROR,
                    text : i18n.PLUGINS_WORKSPACES_MEMBERS_GENERAL_ERROR_TEXT,
                    details: e
                });
            }

        },

        /**
         * Edit member's rights profiles
         * @param {Object} profiles The profiles to set for each module (module id as key, profile id as value)
         * @param {Object} member The member to edit
         */
        async editMemberProfile(profiles, member) {
            try {
                AmetysFront.Event.fire('loaderStart', {
                    text: i18n.PLUGINS_WORKSPACES_MEMBERS_EDIT_MEMBERS_START
                });

                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.members.ProjectMemberManager',
                    methodName: 'setProjectMemberData',
                    parameters: [
                        projectName,
                        member.id,
                        member.isGroup ? 'group' : 'user',
                        profiles,
                        null // No role
                    ]
                })

                if (result.success)
                {
                    AmetysFront.Event.fire('loaderEnd', { 
                        text : i18n.PLUGINS_WORKSPACES_MEMBERS_EDIT_MEMBERS_SUCCESS
                    });
                }
                else if (result["message"] == "unknown-project")
                {
                    AmetysFront.Event.fire('loaderFail', { 
                        title: i18n.PLUGINS_WORKSPACES_MEMBERS_EDIT_MEMBERS_ERROR,
                        text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_PROJECT_ERROR_TEXT,
                        details: null
                    });
                }
                else if (result["message"] == "unknown-group")
                {
                    AmetysFront.Event.fire('loaderFail', { 
                        title: i18n.PLUGINS_WORKSPACES_MEMBERS_EDIT_MEMBERS_ERROR,
                        text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_GROUP_ERROR_TEXT,
                        details: null
                    });
                }
                else if (result["message"] == "unknown-user")
                {
                    AmetysFront.Event.fire('loaderFail', { 
                        title: i18n.PLUGINS_WORKSPACES_MEMBERS_EDIT_MEMBERS_ERROR,
                        text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_USER_ERROR_TEXT,
                        details: null
                    });
                }
                else
                {
                    AmetysFront.Event.fire('loaderFail', { 
                        title: i18n.PLUGINS_WORKSPACES_MEMBERS_EDIT_MEMBERS_ERROR,
                        text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_ERROR_TEXT,
                        details: null
                    });
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_MEMBERS_EDIT_MEMBERS_ERROR,
                    text : i18n.PLUGINS_WORKSPACES_MEMBERS_GENERAL_ERROR_TEXT,
                    details: e
                });
            }
        },

        /**
         * Remove a member
         * @param {Object} member The member to remove
         */
        async removeMember(member) {

            try
            {
                
                AmetysFront.Event.fire('loaderStart', {
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETING
                });

                let result = await callMethod({
                    role: 'org.ametys.plugins.workspaces.members.ProjectMemberManager',
                    methodName: 'removeMember',
                    parameters: [
                        projectName,
                        member.id,
                        member.isGroup ? 'group' : 'user'
                    ]
                })
                if (result.success)
                {
                    const searchUserStore = useSearchUsersStore()

                    const idx = this.members.findIndex(foundMember => foundMember.id == member.id);
                    if (idx > -1)
                    {
                        this.members.splice(idx, 1);
                    }
                    searchUserStore.needReload = true;
                    AmetysFront.Event.fire('loaderEnd', { 
                        text : i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBERS_SUCCESS
                    });
                }
                else
                {
                    if (result["message"] == "unknown-project")
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_ERROR,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_PROJECT_ERROR_TEXT,
                            details: null
                        });
                    }
                    if (result["message"] == "unknown-users")
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_ERROR,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_USER_ERROR_TEXT,
                            details: null
                        });
                    }
                    if (result["message"] == "unknown-group")
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_ERROR,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_UNKNOWN_GROUP_ERROR_TEXT,
                            details: null
                        });
                    }
                    if (result["message"] == "current-user")
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_ERROR,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_CURRENT_USER_ERROR,
                            details: null
                        });
                    }
                    if (result["message"] == "only-manager")
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_ERROR,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_ONLY_MANAGER_ERROR,
                            details: null
                        });
                    }
                    if (result["message"] == "unknown-member")
                    {
                        AmetysFront.Event.fire('loaderFail', { 
                            title: i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_ERROR,
                            text : i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_UNKNOWN_MEMBER_ERROR,
                            details: null
                        });
                    }
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', { 
                    title: i18n.PLUGINS_WORKSPACES_MEMBERS_DELETE_MEMBER_ERROR,
                    text : i18n.PLUGINS_WORKSPACES_MEMBERS_GENERAL_ERROR_TEXT,
                    details: e
                });
            }
        },

        /*
         * Load the profiles and modules and store them
         */
        async loadProfilesAndModules() {
            try
            {
                const { profiles, modules } = await callMethod({
                    role: 'org.ametys.plugins.workspaces.project.rights.ProjectRightHelper',
                    methodName: 'getProjectRightsData',
                    parameters: [ projectName]
                });
                this.profiles = profiles;
                this.modules = modules;
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_MEMBERS_GET_PROJECT_RIGHTS_ERROR,
                    text: i18n.PLUGINS_WORKSPACES_MEMBERS_GENERAL_ERROR_TEXT,
                    details: e
                });
            }
        }
    }
})
