/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.about;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.plugins.workspaces.AbstractWorkspaceModule;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.project.ProjectConstants;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.site.Site;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinTemplate;
import org.ametys.web.skin.SkinTemplateZone;
import org.ametys.web.workflow.CreateContentFunction;

import com.opensymphony.workflow.WorkflowException;

/**
 * Manager for the About module
 */
public class AboutWorkspaceModule extends AbstractWorkspaceModule
{
    /** Avalon ROLE */
    public static final String ABOUT_MODULE_ID = AboutWorkspaceModule.class.getName();
    
    /** Workspaces tasks list node name */
    private static final String __WORKSPACES_ABOUT_NODE_NAME = "about";
    
    /** The initial workflow action id */
    private static final int __INIT_WORKFLOW_ACTION_ID = 16;
    
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
    }
    
    @Override
    public String getId()
    {
        return ABOUT_MODULE_ID;
    }
    
    @Override
    public String getModuleName()
    {
        return __WORKSPACES_ABOUT_NODE_NAME;
    }
    
    public int getOrder()
    {
        return ORDER_ABOUT;
    }
    
    @Override
    protected String getModulePageName()
    {
        return "about";
    }
    
    public I18nizableText getModuleTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_ABOUT_LABEL");
    }
    public I18nizableText getModuleDescription()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_ABOUT_DESCRIPTION");
    }
    @Override
    protected I18nizableText getModulePageTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_WORKSPACE_PAGE_ABOUT_TITLE");
    }
    
    @Override
    protected String getModulePageTemplate()
    {
        return ProjectConstants.ABOUT_TEMPLATE;
    }
    
    @Override
    protected void initializeModulePage(ModifiablePage page)
    {
        String contentTitle = StringUtils.defaultIfEmpty(_i18nUtils.translate(getModulePageTitle(), page.getSitemapName()), "Missing title");
        _initializeAboutDefaultZone(page, page.getSite(), page.getSitemapName(), contentTitle);
    }
    
    /**
     * Initialize the default zone for the about page
     * @param aboutPage The about page
     * @param site The site
     * @param sitemapName The sitemap name
     * @param title The content title
     */
    protected void _initializeAboutDefaultZone(ModifiablePage aboutPage, Site site, String sitemapName, String title)
    {
        ModifiableZone defaultZone = aboutPage.createZone("default");
        
        ModifiableZoneItem defaultZoneItem = defaultZone.addZoneItem();
        defaultZoneItem.setType(ZoneType.CONTENT);
        
        try
        {
            ModifiableWorkflowAwareContent content = createAboutContent(site, sitemapName, title);
            defaultZoneItem.setContent(content);
            content.saveChanges();
        }
        catch (WorkflowException e)
        {
            getLogger().error("Unable to initialize the About page content for the new workspace, the about page will not be editable until the content is manually created in the BackOffice", e);
        }
    }
    
    /**
     * Retrieves the rights for the current user in the project
     * @return The project
     */
    /**
     * Create the about content
     * @param site The site
     * @param sitemapName the name of the sitemap
     * @param title The content title
     * @return The content
     * @throws WorkflowException if an error occurred
     */
    public ModifiableWorkflowAwareContent createAboutContent(Site site, String sitemapName, String title) throws WorkflowException
    {
        Map<String, Object> inputs = new HashMap<>();
        inputs.put(org.ametys.cms.workflow.CreateContentFunction.CONTENT_TITLE_KEY, title);
        inputs.put(org.ametys.cms.workflow.CreateContentFunction.CONTENT_NAME_KEY, NameHelper.filterName(title));
        inputs.put(org.ametys.cms.workflow.CreateContentFunction.CONTENT_TYPES_KEY, new String[] {WorkspacesConstants.ABOUT_CONTENT_TYPE});
        inputs.put(org.ametys.cms.workflow.CreateContentFunction.CONTENT_LANGUAGE_KEY, sitemapName);
        inputs.put(CreateContentFunction.SITE_KEY, site.getName());
        
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow();
        workflow.initialize(WorkspacesConstants.CONTENT_WORKFLOW_NAME, __INIT_WORKFLOW_ACTION_ID, inputs);
    
        @SuppressWarnings("unchecked")
        Map<String, Object> results = (Map<String, Object>) inputs.get(AbstractWorkflowComponent.RESULT_MAP_KEY);
        ModifiableWorkflowAwareContent content = (ModifiableWorkflowAwareContent) results.get(Content.class.getName());
        
        return content;
    }
    
    

    
    /**
     * Initialize the new about page
     * @param aboutPage The page
     * @param site The site
     * @param sitemapName The sitemap name 
     * @param title The page title
     * @param parentPage The parent page
     */
    protected void _initializeAboutPage(ModifiablePage aboutPage, Site site, String sitemapName, String title, Page parentPage)
    {
        Skin skin = _skinsManager.getSkin(site.getSkinId());
        SkinTemplate template = skin.getTemplate(ProjectConstants.ABOUT_TEMPLATE);
        
        if (template != null)
        {
            // Set the type and template.
            aboutPage.setType(PageType.CONTAINER);
            aboutPage.setTemplate(ProjectConstants.ABOUT_TEMPLATE);
            
            // Initialize the zones.
            Map<String, SkinTemplateZone> templateZones = template.getZones();
            if (templateZones.containsKey("default"))
            {
                _initializeAboutDefaultZone(aboutPage, site, sitemapName, title);
            }
            else
            {
                getLogger().error("A 'default' zone is mandatory in the about template!");
                return;
            }
            
            // Tag page as a sitemap section.
            aboutPage.tag("SECTION");
        }
        else
        {
            String errorMsg = String.format(
                    "The project workspace  '%s' was created with the skin '%s'  which doesn't possess the mandatory template '%s'.\nThe '%s' page of the project workspace could not be initialized.",
                    site.getName(), site.getSkinId(), ProjectConstants.ABOUT_TEMPLATE, aboutPage.getName());
            
            getLogger().error(errorMsg);
        }
    }
    
    @Override
    public Set<String> getAllowedEventTypes()
    {
        return Set.of();
    }

    @Override
    protected boolean _showActivatedStatus()
    {
        return false;
    }
}
