/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.SourceUtil;

import org.ametys.core.right.RightManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.notification.preferences.NotificationPreferencesHelper;
import org.ametys.plugins.workspaces.project.notification.preferences.NotificationPreferencesHelper.Frequency;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.WebConstants;
import org.ametys.web.activities.notify.ActivityNotifier;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.site.Site;

/**
 * Abstract class representing a activity notifier for workspaces
 */
public abstract class AbstractWorkspacesActivityNotifier implements ActivityNotifier, Serviceable, PluginAware, Contextualizable
{
    /** The right manager */
    protected RightManager _rightManager;
    
    /** The notification preference helper */
    protected NotificationPreferencesHelper _notificationPreferenceHelper;
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The project manager */
    protected ProjectManager _projectManager;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The rendering context handler */
    protected RenderingContextHandler _renderingContextHandler;
    
    /** The source resolver */
    protected SourceResolver _srcResolver;
    
    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The user languages manager */
    protected UserLanguagesManager _userLanguagesManager;
    
    /** The workspaces helper */
    protected WorkspacesHelper _workspacesHelper;
    
    /** The plugin name */
    protected String _pluginName;

    /** The context */
    protected Context _context;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _notificationPreferenceHelper = (NotificationPreferencesHelper) manager.lookup(NotificationPreferencesHelper.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _userLanguagesManager = (UserLanguagesManager) manager.lookup(UserLanguagesManager.ROLE);
        _workspacesHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public Map<String, List<String>> getUsersToNotifyByLanguage(Activity activity)
    {
        boolean returnAll = Config.getInstance().getValue("runtime.mail.massive.sending");
        
        String projectName = activity.getValue(AbstractWorkspacesActivityType.PROJECT_NAME);
        Project project = _projectManager.getProject(projectName);
        String defaultLanguage = StringUtils.defaultIfBlank(_workspacesHelper.getLang(project), _userLanguagesManager.getDefaultLanguage());
        
        // FIXME GG right should be provided directly by the activity
        AmetysObject aObject = getTargetAmetysObject(activity);
        Set<UserIdentity> readAccessUsers = _rightManager.getReadAccessAllowedUsers(aObject).resolveAllowedUsers(returnAll);
        
        Map<String, List<String>> usersByLanguage =  readAccessUsers.stream()
            .filter(userId -> _notificationPreferenceHelper.askedToBeNotified(userId, activity.getValue(AbstractWorkspacesActivityType.PROJECT_NAME), Frequency.EACH))
            .map(_userManager::getUser)
            .filter(Objects::nonNull)
            .map(user -> Pair.of(user.getLanguage(), user.getEmail()))
            .filter(p -> StringUtils.isNotEmpty(p.getRight()))
            .collect(Collectors.groupingBy(
                    p -> {
                        return StringUtils.defaultIfBlank(p.getLeft(), defaultLanguage);
                    },
                    Collectors.mapping(
                            Pair::getRight,
                            Collectors.toList()
                    )
                )
            );
        
        return usersByLanguage;
    }
    
    /**
     * Retrieve the ametys object targeted by this activity.
     * This method is intended for right computation purposes.
     * @param activity the activity
     * @return the target
     */
    // FIXME this should not exist and the activity itself should provide the right. (probably by a right convertor)
    public abstract AmetysObject getTargetAmetysObject(Activity activity);

    @Override
    public String getMailSubject(Activity activity, String language)
    {
        return _i18nUtils.translate(new I18nizableText("plugin." + _pluginName, _getSubjectI18nKey(activity), getSubjectI18nParams(activity)), language);
    }
    
    /**
     * Get the subject i18n key
     * @param activity the activity
     * @return the subject i18n key
     */
    protected String _getSubjectI18nKey(Activity activity)
    {
        return "PROJECT_MAIL_NOTIFICATION_SUBJECT_" + StringUtils.replaceChars(activity.getEventType().toUpperCase(), '.', '_');
    }

    /**
     * Get the subject i18n parameters
     * @param activity the activity
     * @return the subject i18n parameters
     */
    public List<String> getSubjectI18nParams(Activity activity)
    {
        List<String> i18nparams = new ArrayList<>();
        i18nparams.add(activity.getValue(AbstractWorkspacesActivityType.PROJECT_TITLE)); // {0}
        return i18nparams;
    }
    
    @Override
    public String getMailTextBody(Activity activity, String language)
    {
        // No text body
        return null;
    }
    
    @Override
    public String getMailHtmlBody(Activity activity, String language)
    {
        String projectName = activity.getValue(AbstractWorkspacesActivityType.PROJECT_NAME);
        Project project = _projectManager.getProject(projectName);
        Site site = project.getSite();
        
        String mailBody;
        Source source = null;
        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
        
        try
        {
            // Force rendering context.FRONT to resolve URI
            _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
            
            Request request = ContextHelper.getRequest(_context);
            
            request.setAttribute("lang", language);
            request.setAttribute("sitemapLanguage", language);
            request.setAttribute("forceAbsoluteUrl", true);
            request.setAttribute(WebConstants.REQUEST_ATTR_SITE, site);
            request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, site.getName());
            request.setAttribute(WebConstants.REQUEST_ATTR_SKIN_ID, site.getSkinId());
            
            source = _srcResolver.resolveURI(getMailBodyURI(activity), null, Map.of(AbstractWorkspacesActivityType.ACTIVITY_CONTEXT_PARAM, activity));
            
            try (InputStream is = source.getInputStream())
            {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                SourceUtil.copy(is, bos);
                
                mailBody = bos.toString("UTF-8");
            }
        }
        catch (IOException e)
        {
            throw new RuntimeException("Failed to create mail body", e);
        }
        finally
        {
            _renderingContextHandler.setRenderingContext(currentContext);
            
            if (source != null)
            {
                _srcResolver.release(source);
            }
        }
        return mailBody;
    }
    
    /**
     * Get the URI to resolve to get the mail body
     * @param activity the activity
     * @return the uri
     */
    public abstract String getMailBodyURI(Activity activity);
    
}
