/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.activities.AbstractActivityType;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Class representing a generic activity type for workspaces
 */
public abstract class AbstractWorkspacesActivityType extends AbstractActivityType implements Contextualizable
{
    /** The name used to store the activity in the context parameter when generating the notification */
    public static final String ACTIVITY_CONTEXT_PARAM = "activity";
    
    /** Constant for project's name JSON key. The value is a {@link String}. */
    public static final String PROJECT_NAME = "projectName";
    /** Constant for project's title JSON key. The value is a {@link String}. */
    public static final String PROJECT_TITLE = "projectTitle";
    /** Constant for project's access JSON key. The value is a boolean. */
    public static final String PROJECT_ACCESS = "hasProjectAccess";
    /** Constant for project's url JSON key. The value is a {@link String}. */
    public static final String PROJECT_URL = "projectUrl";
    /** Constant for project's category JSON key. The value is a {@link String}. */
    public static final String PROJECT_CATEGORY = "category";
    
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The project member manager */
    protected ProjectMemberManager _projectMemberManager;
    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;
    /** The User manager */
    protected UserManager _userManager;
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The context */
    protected Context _context;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _projectManager = (ProjectManager) serviceManager.lookup(ProjectManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) serviceManager.lookup(ProjectMemberManager.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Add the value of item if exists to JSON activity representation
     * @param json the JSON activity
     * @param activity the activity 
     * @param dataPath the data path
     * @param key the JSON key
     */
    protected void _addValueIfExists(Map<String, Object> json, Activity activity, String dataPath, String key)
    {
        if (activity.hasValue(dataPath))
        {
            json.put(key, activity.getValue(dataPath));
        }
    }
    
    @Override
    public void setAdditionalActivityData(Activity activityNode, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activityNode, parameters);
        
        activityNode.setValue(PROJECT_NAME, parameters.get(PROJECT_NAME));
        activityNode.setValue(PROJECT_TITLE, parameters.get(PROJECT_TITLE));
    }
    
    @Override
    public Map<String, Object> additionnalDataToJSONForClient(Activity activity)
    {
        Map<String, Object> json = super.additionnalDataToJSONForClient(activity);
        
        // Project
        String projectName = activity.getValue(PROJECT_NAME);
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            boolean hasAccess = _projectMemberManager.isProjectMember(project, activity.getAuthor());
            json.put(PROJECT_ACCESS, hasAccess);
            String siteUrl = _projectManager.getProjectUrl(project, null);
            if (siteUrl != null)
            {
                json.put(PROJECT_URL, siteUrl);
            }
        }
        
        return json;
    }
    
    @Override
    public boolean isMergeable(Activity activity1, Activity activity2)
    {
        if (!super.isMergeable(activity1, activity2))
        {
            return false;
        }
        String project1 = activity1.getValue(PROJECT_NAME);
        String project2 = activity2.getValue(PROJECT_NAME);
        
        return project1 != null && project2 != null && project1.equals(project2);
    }
    
    /**
     * Helper method to retrieve the parent project of an ametys object
     * @param ao the ametys object
     * @return the parent project or null if the object doesn't have have project as parent
     */
    protected Project getParentProject(AmetysObject ao)
    {
        AmetysObject parent = ao.getParent();
        if (parent == null)
        {
            return null;
        }
        else if (parent instanceof Project project)
        {
            return project;
        }
        else
        {
            return getParentProject(parent);
        }
    }
    
    /**
     * Get the expression to filter activity on pattern
     * @param pattern the pattern expression
     * @return the pattern expression
     */
    public abstract Expression getFilterPatternExpression(String pattern);
    
    @Override
    public List<ActivityArguments> getActivitiesArguments(Event event)
    {
        Project project = getProjectFromEvent(event);
        if (project != null)
        {
            return List.of(new ActivityArguments(project, _getProjectParameters(project)));
        }
        
        return List.of();
    }
    
    /**
     * Get the project parameters to create the activity
     * @param project the project
     * @return the project parameters
     */
    protected Map<String, Object> _getProjectParameters(Project project)
    {
        Map<String, Object> params = new HashMap<>();
        params.put(AbstractWorkspacesActivityType.PROJECT_NAME, project.getName());
        params.put(AbstractWorkspacesActivityType.PROJECT_TITLE, project.getTitle());
        
        return params;
    }

    /**
     * Get the project from the event
     * @param event the event
     * @return the project
     */
    public abstract Project getProjectFromEvent(Event event);
    
}
