/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.projects;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.GroupManager;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityNotifier;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.notification.preferences.NotificationPreferencesHelper.Frequency;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Class representing a activity notifier for member added in workspaces
 */
public class MemberAddedActivityNotifier extends AbstractWorkspacesActivityNotifier
{
    /** The group manager */
    protected GroupManager _groupManager;
    
    /** The project member manager */
    protected ProjectMemberManager _projectMemberManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _groupManager = (GroupManager) manager.lookup(GroupManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
    }
    
    public boolean supports(ActivityType activityType)
    {
        return activityType instanceof MemberAddedActivityType;
    }
    
    @Override
    public Map<String, List<String>> getUsersToNotifyByLanguage(Activity activity)
    {
        Stream<User> previousUser = null;
        Project project = _projectManager.getProject(activity.getValue(AbstractWorkspacesActivityType.PROJECT_NAME));

        // if the new member is a user, we just need to inform all other user
        // we have already check that it is in fact a new member
        MemberType memberType = MemberType.valueOf(activity.<String> getValue(MemberAddedActivityType.MEMBER_TYPE).toUpperCase());
        if (MemberType.USER.equals(memberType))
        {
            UserIdentity userIdentity = UserIdentity.stringToUserIdentity(activity.getValue(MemberAddedActivityType.MEMBER));
            if (userIdentity != null)
            {
                User newUser = _userManager.getUser(userIdentity);
                if (newUser != null)
                {
                    previousUser = _projectMemberManager.getProjectMembers(project, true).stream().map(ProjectMember::getUser).filter(u -> !u.equals(newUser));
                }
            }
        }
        else
        {
            GroupIdentity groupIdentity = GroupIdentity.stringToGroupIdentity(activity.getValue(MemberAddedActivityType.MEMBER));
            if (groupIdentity != null)
            {
                Group group = _groupManager.getGroup(groupIdentity);
                if (group != null)
                {
                    previousUser = _projectMemberManager.getProjectMembers(project, false).stream()
                            // we remove the new group from the project members.
                            .filter(member -> MemberType.USER.equals(member.getType()) || group.equals(member.getGroup()))
                            // now we can expand all the group. If a user from
                            // the new group was already present he shall
                            // receive a notification as an activity was created
                            .flatMap(member -> {
                                if (MemberType.USER.equals(member.getType()))
                                {
                                    return Stream.<User> of(member.getUser());
                                }
                                else
                                {
                                    return member.getGroup().getUsers().stream().map(_userManager::getUser);
                                }
                            });
                }
            }
        }
        if (previousUser != null)
        {
            String defaultLanguage = _userLanguagesManager.getDefaultLanguage();
            return previousUser.filter(user -> _notificationPreferenceHelper.askedToBeNotified(user.getIdentity(), activity.getValue(AbstractWorkspacesActivityType.PROJECT_NAME),
                    Frequency.EACH))
                    .map(user -> Pair.of(user.getLanguage(), user.getEmail()))
                    .filter(p -> StringUtils.isNotEmpty(p.getRight()))
                    .collect(Collectors.groupingBy(
                            p -> {
                                return StringUtils.defaultIfBlank(p.getLeft(), defaultLanguage);
                            },
                            Collectors.mapping(
                                    Pair::getRight,
                                    Collectors.toList()
                            )
                        )
                    );
        }
        else
        {
            // Something bad happened…
            return Map.of();
        }
    }
    
    
    @Override
    public String getMailBodyURI(Activity activity)
    {
        return "cocoon://_plugins/workspaces/notification-mail-member";
    }
    
    @Override
    public AmetysObject getTargetAmetysObject(Activity activity)
    {
        // we don't need this method as we override the AbstractWorkspacesActivityNotifier::getUsersEmailToNotify method
        return null;
    }
}
